<?php
   /**
   * Session.php
   * 
   * The Session class is meant to simplify the task of keeping
   * track of logged in users and also guests.
   *
   */
   require_once('database.php');
   require_once('form.php');
   require_once('mailer.php');
   
   class Session {
      var $username;            // Username given on sign-up
      var $userid;              // Random value generated on current login
      var $userlevel;           // The level to which the user pertains
      var $time;                // Time user was last active (page loaded)
      var $logged_in;           // True if user is logged in, false otherwise
      var $userinfo = array();  // The array holding all user info
      var $url;                 // The page url current being viewed
      var $referrer;            // Last recorded site page viewed
      /**
      * Note: referrer should really only be considered the actual
      * page referrer in process.php, any other time it may be
      * inaccurate.
      */
   
      /* Class constructor */
      public function __construct() {
         $this->time = time();
         $this->startSession();
      }
      
      /**
      * startSession - Performs all the actions necessary to 
      * initialize this session object. Tries to determine if the
      * the user has logged in already, and sets the variables 
      * accordingly. Also takes advantage of this page load to
      * update the active visitors tables.
      */
      public function startSession() {
         global $database;  // The database connection
         session_start();   // Tell PHP to start the session
         
         /* Determine if user is logged in */
         $this->logged_in = $this->checkLogin();
         
         /**
         * Set guest value to users not logged in, and update
         * active guests table accordingly.
         */
         if (!$this->logged_in) {
            $this->username = $_SESSION['username'] = GUEST_NAME;
            $this->userlevel = GUEST_LEVEL;
            $database->addActiveGuest($_SERVER['REMOTE_ADDR'], $this->time);
         }
         /* Update users last active timestamp */
         else {
            $database->addActiveUser($this->username, $this->time);
         }
         
         /* Remove inactive visitors from database */
         $database->removeInactiveUsers();
         $database->removeInactiveGuests();
         
         /* Set referrer page */
         if (isset($_SESSION['url']))
            $this->referrer = $_SESSION['url'];
         else
            $this->referrer = "/";
         
         /* Set current url */
         $this->url = $_SESSION['url'] = $_SERVER['PHP_SELF'];
      }
      
      /**
      * checkLogin - Checks if the user has already previously
      * logged in, and a session with the user has already been
      * established. Also checks to see if user has been remembered.
      * If so, the database is queried to make sure of the user's 
      * authenticity. Returns true if the user has logged in.
      */
      public function checkLogin() {
         global $database;  // The database connection
         
         /* Check if user has been remembered */
         if (isset($_COOKIE['cookname']) && isset($_COOKIE['cookid'])) {
            $this->username = $_SESSION['username'] = $_COOKIE['cookname'];
            $this->userid   = $_SESSION['userid']   = $_COOKIE['cookid'];
         }
         
         /* Username and userid have been set and not guest */
         if (isset($_SESSION['username']) && isset($_SESSION['userid']) &&
                   $_SESSION['username'] != GUEST_NAME) {
            /* Confirm that username and userid are valid */
            if ($database->confirmUserID($_SESSION['username'], $_SESSION['userid']) != 0) {
               /* Variables are incorrect, user not logged in */
               unset($_SESSION['username']);
               unset($_SESSION['userid']);
               
               return false;
            }
            
            /* User is logged in, set class variables */
            $this->userinfo  = $database->getUserInfo($_SESSION['username']);
            $this->username  = $this->userinfo['username'];
            $this->userid    = $this->userinfo['userid'];
            $this->userlevel = $this->userinfo['userlevel'];
            
            return true;
         }
         /* User not logged in */
         else {
            return false;
         }
      }
      
      /**
      * login - The user has submitted his username and password
      * through the login form, this function checks the authenticity
      * of that information in the database and creates the session.
      * Effectively logging in the user if all goes well.
      */
      public function login($subuser, $subpass, $subremember) {
         require_once('_lang/'.PROG_LANG.'.php');
         global $database, $form;  // The database and form object
         
         /* Username error checking */
         $field = "user";  // Use field name for username
         if (!$subuser || strlen($subuser = trim($subuser)) == 0) {
            $form->setError($field, $STD_010);
         }
         else {
            /* Check if username is not alphanumeric */
            if (!preg_match("/^([áéíóöőúüűÁÉÍÓÖŐÚÜŰ0-9.a-zA-Z\s])*$/", $subuser))
               $form->setError($field, $LNG_091);
         }
         
         /* Password error checking */
         $field = "pass";  // Use field name for password
         if (!$subpass)
            $form->setError($field, $LNG_092);
         
         /* Return if form errors exist */
         if ($form->num_errors > 0)
            return false;
         
         /* Checks that username is in database and password is correct */
         $subuser = stripslashes($subuser);
         $result = $database->confirmUserPass($subuser, md5($subpass));
         
         /* Check error codes */
         if ($result == 1) {
            $field = "user";
            $form->setError($field, $LNG_093);
         }
         else if($result == 2) {
            $field = "pass";
            $form->setError($field, $LNG_094);
         }
         
         /* Return if form errors exist */
         if ($form->num_errors > 0) {
            return false;
         }
         
         /* Username and password correct, register session variables */
         $this->userinfo  = $database->getUserInfo($subuser);
         $this->username  = $_SESSION['username'] = $this->userinfo['username'];
         $this->userid    = $_SESSION['userid']   = $this->generateRandID();
         $this->userlevel = $this->userinfo['userlevel'];
         
         /* Insert userid into database and update active users table */
         $database->updateUserField($this->username, "userid", $this->userid);
         $database->addActiveUser($this->username, $this->time);
         $database->removeActiveGuest($_SERVER['REMOTE_ADDR']);
         
         /**
         * This is the cool part: the user has requested that we remember that
         * he's logged in, so we set two cookies. One to hold his username,
         * and one to hold his random value userid. It expires by the time
         * specified in constants.php. Now, next time he comes to our site, we will
         * log him in automatically, but only if he didn't log out before he left.
         */
         if ($subremember) {
            setcookie("cookname", $this->username, time()+COOKIE_EXPIRE, COOKIE_PATH);
            setcookie("cookid",   $this->userid,   time()+COOKIE_EXPIRE, COOKIE_PATH);
         }
         
         /* Login completed successfully */
         return true;
      }
      
      /**
      * logout - Gets called when the user wants to be logged out of the
      * website. It deletes any cookies that were stored on the users
      * computer as a result of him wanting to be remembered, and also
      * unsets session variables and demotes his user level to guest.
      */
      public function logout() {
         global $database;  // The database connection
         
         /**
         * Delete cookies - the time must be in the past,
         * so just negate what you added when creating the
         * cookie.
         */
         if (isset($_COOKIE['cookname']) && isset($_COOKIE['cookid'])) {
            setcookie("cookname", "", time()-COOKIE_EXPIRE, COOKIE_PATH);
            setcookie("cookid",   "", time()-COOKIE_EXPIRE, COOKIE_PATH);
         }
         
         /* Unset PHP session variables */
         unset($_SESSION['username']);
         unset($_SESSION['userid']);
         
         /* Reflect fact that user has logged out */
         $this->logged_in = false;
         
         /**
         * Remove from active users table and add to
         * active guests tables.
         */
         $database->removeActiveUser($this->username);
         $database->addActiveGuest($_SERVER['REMOTE_ADDR'], $this->time);
         
         /* Set user level to guest */
         $this->username  = GUEST_NAME;
         $this->userlevel = GUEST_LEVEL;
      }
      
      /**
      * register - Gets called when the user has just submitted the
      * registration form. Determines if there were any errors with
      * the entry fields, if so, it records the errors and returns
      * 1. If no errors were found, it registers the new user and
      * returns 0. Returns 2 if registration failed.
      */
      public function register($subuser, $subpass, $subemail) {
         require_once('_lang/'.PROG_LANG.'.php');
         global $database, $form, $mailer;  // The database, form and mailer objects
         
         /* Username error checking */
         $field = "user";  //Use field name for username
         if (!$subuser || strlen($subuser = trim($subuser)) == 0) {
            $form->setError($field, $STD_010);
         }
         else {
            /* Spruce up username, check length */
            $subuser = stripslashes($subuser);
            if (strlen($subuser) < 5)
               $form->setError($field, $LNG_095);
            else if (strlen($subuser) > 30)
               $form->setError($field, $LNG_096);
            /* Check if username is not alphanumeric */
            else if (!preg_match("/^([áéíóöőúüűÁÉÍÓÖŐÚÜŰ0-9.a-zA-Z\s])+$/", $subuser))
               $form->setError($field, $LNG_091);
            /* Check if username is reserved */
            else if (strcasecmp($subuser, GUEST_NAME) == 0)
               $form->setError($field, $LNG_097);
            /* Check if username is already in use */
            else if ($database->usernameTaken($subuser)) {
               //unset($_SESSION['username']);
               //unset($_SESSION['userid']);
               $form->setError($field, $LNG_098);
            }
            /* Check if username is banned */
            else if ($database->usernameBanned($subuser))
               $form->setError($field, $LNG_099);
         }
         
         /* Password error checking */
         $field = "pass";  //Use field name for password
         if (!$subpass) {
            $form->setError($field, $LNG_092);
         }
         else {
            /* Spruce up password and check length*/
            $subpass = stripslashes($subpass);
            if (strlen($subpass) < 4)
               $form->setError($field, $LNG_100);
            /* Check if password is not alphanumeric */
            else if (!preg_match("/^([0-9._a-zA-Z])+$/", ($subpass = trim($subpass))))
               $form->setError($field, $LNG_101);
            /**
            * Note: I trimmed the password only after I checked the length
            * because if you fill the password field up with spaces
            * it looks like a lot more characters than 4, so it looks
            * kind of stupid to report "password too short".
            */
         }
         
         /* Email error checking */
         $field = "email";  // Use field name for email
         if (!$subemail || strlen($subemail = trim($subemail)) == 0) {
            $form->setError($field, $LNG_102);
         }
         else {
            /* Check if valid email address */
            $regex = "/^[_+a-z0-9-]+(\.[_+a-z0-9-]+)*"
                    ."@[a-z0-9-]+(\.[a-z0-9-]{1,})*"
                    ."\.([a-z]{2,}){1}$/";
            if (!preg_match($regex,$subemail)) {
               $form->setError($field, $LNG_103.$subemail.$regex);
            }
            $subemail = stripslashes($subemail);
         }
         
         /* Errors exist, have user correct them */
         if ($form->num_errors > 0) {
            return 1;  // Errors with form
         }
         /* No errors, add the new account to the */
         else {
            if ($database->addNewUser($subuser, md5($subpass), $subemail)) {
               if (EMAIL_WELCOME)
                  $mailer->sendWelcome($subuser,$subemail,$subpass);
               return 0;  // New user added succesfully
            }
            else {
               return 2;  // Registration attempt failed
            }
         }
      }
      
      /**
      * editAccount - Attempts to edit the user's account information
      * including the password, which it first makes sure is correct
      * if entered, if so and the new password is in the right
      * format, the change is made. All other fields are changed
      * automatically.
      */
      public function editAccount($subcurpass, $subnewpass, $subemail) {
         require_once('_lang/'.PROG_LANG.'.php');
         global $database, $form;  // The database and form object
         
         /* Current password has been entered */
         if ($subcurpass) {
            $field = "curpass";
            /* Check if the current password is too short or contains A-Z and/or a-z and/or 0-9 characters */
            $subcurpass = stripslashes($subcurpass);
            if (strlen($subcurpass) < 4 ||
               !preg_match("/^([A-Z.a-z0-9])+$/", ($subcurpass = trim($subcurpass)))) {
               $form->setError($field, $LNG_105);
            }
            
            /* Current password entered is incorrect because of its MD5 value*/
            if ($database->confirmUserPass($this->username,md5($subcurpass)) != 0) {
               $form->setError($field, $LNG_106);
            }
            else {
               /* New password has been entered */
               if ($subnewpass) {
                  /* New Password error checking */
                  $field = "newpass";  // Use field name for new password
                  /* Spruce up password and check length*/
                  $subpass = stripslashes($subnewpass);
                  if (strlen($subnewpass) < 4) {
                     $form->setError($field, $LNG_107);
                  }
                  /* Check if the new password is not alphanumeric */
                  else if (!preg_match("/^([A-Z.a-z0-9])+$/", ($subnewpass = trim($subnewpass)))) {
                     $form->setError($field, $LNG_108);
                  }
                  // Check e-mail
                  else {
                     /* Email error checking */
                     $field = "email";  // Use field name for email
                     if ($subemail && strlen($subemail = trim($subemail)) > 0) {
                        /* Check if valid email address */
                        $regex = "^[_+a-z0-9-]+(\.[_+a-z0-9-]+)*"
                                 ."@[a-z0-9-]+(\.[a-z0-9-]{1,})*"
                                 ."\.([a-z]{2,}){1}$";
                        if (!preg_match("/$regex/",$subemail)) {
                           $form->setError($field, $LNG_110);
                        }
                        $subemail = stripslashes($subemail);
                     }
                  }
               }
               /* New password has not been entered */
               else {
                  $field = "newpass";  // Use field name for new password
                  $form->setError($field, $LNG_111);
               }
            }
         
            /* Errors exist, have user correct them */
            if ($form->num_errors > 0)
               return false;  //Errors with form
            
            /* Change Email */
            if ($subemail)
               $database->updateUserField($this->username,"email",$subemail);
            
            /* Update password since there were no errors */
            if ($subcurpass && $subnewpass)
               $database->updateUserField($this->username,"password",md5($subnewpass));
            
            /* Success! */
            return true;
         }
         /* Current password has not been entered */
         else {
            $field = "curpass";
            $form->setError($field, $LNG_112);
         }
         
         return false;
      }
      
      /**
      * isAdmin - Returns true if currently logged in user is
      * an administrator, false otherwise.
      */
      public function isAdmin() {
         return ($this->userlevel == ADMIN_LEVEL ||
                 $this->username  == ADMIN_NAME);
      }
      
      /**
      * generateRandID - Generates a string made up of randomized
      * letters (lower and upper case) and digits and returns
      * the md5 hash of it to be used as a userid.
      */
      public function generateRandID() {
         return md5($this->generateRandStr(16));
      }
      
      /**
      * generateRandStr - Generates a string made up of randomized
      * letters (lower and upper case) and digits, the length
      * is a specified parameter.
      */
      public function generateRandStr($length) {
         $randstr = "";
         for ($i=0; $i<$length; $i++) {
            $randnum = mt_rand(0,61);
            if ($randnum < 10)
               $randstr .= chr($randnum+48);
            else if ($randnum < 36)
               $randstr .= chr($randnum+55);
            else
               $randstr .= chr($randnum+61);
         }
         
         return $randstr;
      }
      
      /**
      * getRealIpAddr - Get back the real IP address
      */
      public function getRealIpAddr() {
         if (isSet($_SERVER)) {
            if (isSet($_SERVER["HTTP_X_FORWARDED_FOR"])) {
               $realip = $_SERVER["HTTP_X_FORWARDED_FOR"];
            }
            elseif (isSet($_SERVER["HTTP_CLIENT_IP"])) {
               $realip = $_SERVER["HTTP_CLIENT_IP"];
            }
            else {
               $realip = $_SERVER["REMOTE_ADDR"];
            } 
         }
         else {
            if ( getenv( 'HTTP_X_FORWARDED_FOR' ) ) {
               $realip = getenv( 'HTTP_X_FORWARDED_FOR' );
            }
            elseif ( getenv( 'HTTP_CLIENT_IP' ) ) { 
               $realip = getenv( 'HTTP_CLIENT_IP' );
            }
            else {
               $realip = getenv( 'REMOTE_ADDR' );
            }
         }
         return $realip;
      }
      
      /**
      * i2c_realip - Returns the real IP address of the user
      */
      public function i2c_realip() {
         // No IP found (will be overwritten by for
         // if any IP is found behind a firewall)
         $ip = FALSE;
         
         // If HTTP_CLIENT_IP is set, then give it priority
         if (!empty($_SERVER["HTTP_CLIENT_IP"])) {
            $ip = $_SERVER["HTTP_CLIENT_IP"];
         }
         
         // User is behind a proxy and check that we discard RFC1918 IP addresses
         // if they are behind a proxy then only figure out which IP belongs to the
         // user.  Might not need any more hackin if there is a squid reverse proxy
         // infront of apache.
         if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            // Put the IP's into an array which we shall work with shortly.
            $ips = explode (", ", $_SERVER['HTTP_X_FORWARDED_FOR']);
            if ($ip)
               array_unshift($ips, $ip); $ip = FALSE;
            
            for ($i = 0; $i < count($ips); $i++) {
               // Skip RFC 1918 IP's 10.0.0.0/8, 172.16.0.0/12 and
               // 192.168.0.0/16
               if (!preg_match('/^(?:10|172\.(?:1[6-9]|2\d|3[01])|192\.168)\./', $ips[$i])) {
                  if (version_compare(phpversion(), "5.0.0", ">=")) {
                     if (ip2long($ips[$i]) != false) {
                        $ip = $ips[$i];
                        break;
                     }
                  }
                  else {
                     if (ip2long($ips[$i]) != -1) {
                         $ip = $ips[$i];
                         break;
                     }
                  }
               }
            }
         }
         
         // Return with the found IP or the remote address
         return ($ip ? $ip : $_SERVER['REMOTE_ADDR']);
      }
      
      /**
       * Get the MySQL/MariaDB character set
      */
      public function ReadDBCharacterset() {
         global $database;
         
         return $database->ReadCharacterSet();
      }
      
      /**
       * createNewBlog - Try to save a new blog of a registered user
       * into the database      
      */
      public function createNewBlog($subcuruser, $subnewtitle, $subnewcontent, $subblognote, $subnewlink, $subnewimage) {
         require_once('_lang/'.PROG_LANG.'.php');
         global $database, $form;  // The database and form object
         
         $subcuruser = stripslashes($subcuruser);
         $subnewtitle = stripslashes($subnewtitle);
         $subnewcontent = stripslashes($subnewcontent);
         $subnewlink = stripslashes($subnewlink);
         $subnewimage = stripslashes($subnewimage);
         
         /* Check if the current blog title is ok */
         if (strlen($subnewtitle) > 0) {
            $field = "blog_title";
            if (strlen($subnewtitle) < 4) {
               $form->setError($field, $LNG_BLOG_07);
            }
            else {
               /* Check if the current blog comment is ok */
               if (strlen($subnewcontent) > 0) {
                  if (strlen($subnewcontent) < 4) {
                     $field = "blog_comment";
                     $form->setError($field, $LNG_BLOG_09);
                  }
               }
               else {
                  $field = "blog_comment";
                  $form->setError($field, $LNG_BLOG_08);
               }
            }
         }
         /* The blog title is empty */
         else {
            if (strlen($subnewtitle) == 0) {
               $field = "blog_title";
               $form->setError($field, $LNG_BLOG_06);
            }
            if (strlen($subnewcontent) == 0) {
               $field = "blog_comment";
               $form->setError($field, $LNG_BLOG_08);
            }
            
         }
         
         /* Errors exist, have user correct them */
         if ($form->num_errors > 0) {
            return 1;  // Errors with form
         }
         /* No errors, add the new blog to the */
         else {
            $subtimestamp = time();
            //$subip=$this->getRealIpAddr();
            $subip=$this->i2c_realip();
            
            if ($database->addNewBlog($subcuruser, $subtimestamp, $subip, $subnewtitle, $subnewcontent, $subblognote, $subnewlink, $subnewimage)) {
               return 0;  // New blog added succesfully
            }
            else {
               return 2;  // Creating a neww blog attempt failed
            }
         }
      }
      
      public function readSavedBlog($currCommand) {
         global $database;  // The database and form object
         
         $blog_result=$database->readSavedBlog($currCommand);
         return $blog_result;
      }
      
      /**
       * uploadImage - upload an image into the uploads directory
      */
      public function uploadNewImage($tmp_dir, $target, $file_name) {
         require_once('_lang/'.PROG_LANG.'.php');
         global $form;
         
         if ($file_name=='') {
            $field = "blog_image_error";
            $form->setError($field, $LNG_113);
         }
         else {
            // image or its extension is not good
            if ($file_name=='' || !preg_match("/(gif|jpe?g|png)$/i", $file_name)) {
               $field = "blog_image_error";
               $form->setError($field, $LNG_114);
            }
            else {
               /* THE FILE WILL BE MOVED FROM THE TEMPORARLY DIRECTRY INTO THE UPLOAD ONE */
               move_uploaded_file($tmp_dir,$target.$file_name);
               $field = "blog_image_correct";
               $form->setCorrect($field, $LNG_115);
               
               return 0;
            }
         }
         
         if ($form->num_errors > 0)
            return 1;  //Errors with form
      }
   };
      
   /**
   * Initialize session object - This must be initialized before
   * the form object because the form uses session variables,
   * which cannot be accessed unless the session has started.
   */
   $session = new Session();
   
   /* Initialize form object */
   $form = new Form();
?>
