<?php
  class SessionDbManagerTool {
    var $dbHost;
    var $dbUser;
    var $dbPasswd;
    var $dbDatabaseName;
    var $salt;
    
    public function __construct($dbhost,$dbuser,$dbpasswd,$dbDBname) {
      $this->dbHost=$dbhost;
      $this->dbUser=$dbuser;
      $this->dbPasswd=$dbpasswd;
      $this->dbDatabaseName=$dbDBname;
      $this->salt=$this->generateSalt();
      
      // set our custom session functions
      session_set_save_handler(
        array($this, 'open'),
        array($this, 'close'),
        array($this, 'read'),
        array($this, 'write'),
        array($this, 'destroy'),
        array($this, 'gc')
      );
      
      // this line prevents unexpected effects when using objects as save handlers
      register_shutdown_function('session_write_close');
    }
    
		/* http://stackoverflow.com/questions/3770150/php-make-session-expire-after-x-minutes */
    /* http://stackoverflow.com/questions/520237/how-do-i-expire-a-php-session-after-30-minutes */
		/* check wheter the current session has been expired */
		public static function CheckSessionExpiration($lastActivityName,$createdName,$lastActivityTimeStamp,
																									$uidName='userID',$uid='',$name,$limit,$path,$domain) {
			/* if the session has been created and the session has been expired after last request
				 then the session will be destroyed */
			if (isset($_SESSION[$lastActivityName])&&(time()-$_SESSION[$lastActivityName]>$lastActivityTimeStamp)) {
				/* unset $_SESSION variable for the run-time */
				session_unset();
				/* destroy session data in storage */
				session_destroy();
				/* create the new session with standard parameters */
				self::InitSession($name,$limit,$path,$domain);
			}
			/* update always last activity time stamp */
			$_SESSION[$lastActivityName]=time();
			
			/* if the session created time has not been created then it will do it */
			if (!isset($_SESSION[$createdName])) {
				/* create in the session the userID */
				if (!isset($_SESSION[$uidName]))
					$_SESSION[$uidName]=$uid;
				/* create the session start time stamp in the session */
				$_SESSION[$createdName]=time();
			}
			/* if the current session has been expired and is active then it will be regenerated */
			else if (time()-$_SESSION[$createdName]>$lastActivityTimeStamp) {
				/* change session ID for the current session an invalidate old session ID */
				session_regenerate_id(true);
				/* update creation time */
				$_SESSION[$createdName]=time();
			}
		}
		
		/* add a new parameter and a new value to a session */
    public function StartSession($session_name, $secure) {
      // Make sure the session cookie is not accessable via javascript.
      $httponly = true;
      // Hash algorithm to use for the sessionid. (use hash_algos() to get a list of available hashes.)
      $session_hash = 'sha512';
      // Check if hash is available
      if (in_array($session_hash, hash_algos())) {
        // Set the has function.
        ini_set('session.hash_function', $session_hash);
      }
      // How many bits per character of the hash.
      // The possible values are '4' (0-9, a-f), '5' (0-9, a-v), and '6' (0-9, a-z, A-Z, "-", ",").
      ini_set('session.hash_bits_per_character', 5);
      // Force the session to only use cookies, not URL variables.
      ini_set('session.use_only_cookies', 1);
      // Get session cookie parameters 
      $cookieParams = session_get_cookie_params(); 
      // Set the parameters
      session_set_cookie_params($cookieParams["lifetime"], $cookieParams["path"], $cookieParams["domain"], $secure, $httponly); 
      // Change the session name 
      session_name($session_name);
      // Now we cat start the session
      session_start();
      // This line regenerates the session and delete the old one. 
      // It also generates a new encryption key in the database. 
      session_regenerate_id(true);    
    }
    
    /* returns with the current name of a session */
    public static function GetSessionName() {
      return (session_name());
    }
    
    /* returns with the current name of a session */
    public static function GetSessionId() {
      return (session_id());
    }
    
    /* this function will be called by the PHP sessions when we start a new session, we use it to start a new database connection */
    function open() {
      if (!function_exists('mysqli_init') && !extension_loaded('mysqli')) {
        die(__METHOD__.': MySQLi extension is not supported!');
      }
      $mysqli=@new MySQLi($this->dbHost,$this->dbUser,$this->dbPasswd,$this->dbDatabaseName);
      if (mysqli_connect_errno()) {
        printf(__METHOD__.": Connect failed: %s", mysqli_connect_error().'<br />');
        echo 'Your parameters (host,username,database): '.$this->dbHost.', '.$this->dbUser.', '.$this->dbDatabaseName;
        exit();
      }
      $this->db = $mysqli;
      
      return true;
    }
    
    /* this function will be called when the sessions want to be closed */
    function close() {
      $this->db->close();
      
      return true;
    }
    
    /**
     * this function will be called by PHP when we try to access a session
     * for example when we use echo $_SESSION['something'];
     * because there might be many calls to this function on a single page,
     * we take advantage of prepared statements, not only for security but
     * for performance also
     * we only prepare the statement once then we can execute it many times
     * we also decrypt the session data that is encrypted in the database
     * we are using 256-bit AES encryption in our sessions
    */
    function read($id) {
      $query="";
			
			if(!isset($this->read_stmt)) {
        $query="SELECT data FROM ".$this->dbDatabaseName." WHERE id = ? LIMIT 1";
				$this->read_stmt = $this->db->prepare($query);
      }
      $this->read_stmt->bind_param('s', $id);
      $this->read_stmt->execute();
      $this->read_stmt->store_result();
      $this->read_stmt->bind_result($data);
      $this->read_stmt->fetch();
      $key = $this->getkey($id);
      $data = $this->decrypt($data, $key);
      
      return $data;
    }
    
    /**
     * this function is used when we assign a value to a session,
     * for example:
     * $_SESSION['something'] = 'something else';
     * the function encrypts all the data which gets inserted
     * into the database
    */
    function write($id, $data) {
      $query="";
			
			// Get unique key
      $key = $this->getkey($id);
      // Encrypt the data
      $data = $this->encrypt($data, $key);
      
      $time = time();
      if (!isset($this->w_stmt)) {
				$query="REPLACE INTO ".$this->dbDatabaseName." (id, set_time, data, session_key) VALUES (?, ?, ?, ?)";
        $this->w_stmt = $this->db->prepare($query);
      }
      
      $this->w_stmt->bind_param('siss', $id, $time, $data, $key);
      $this->w_stmt->execute();
      
      return true;
    }
    
    /* this function deletes the session from the database, it is used by php when we call functions like session__destroy(); */
    function destroy($id) {
      $query="";
			
			if (!isset($this->delete_stmt)) {
        $query="DELETE FROM ".$this->dbDatabaseName." WHERE id = ?";
				$this->delete_stmt = $this->db->prepare($query);
      }
      $this->delete_stmt->bind_param('s', $id);
      $this->delete_stmt->execute();
      
      return true;
    }
    
    /**
     * this function is the garbage collecter function
     * it is called to delete old sessions
     * the frequency in which this function is called
     * is determined by two configuration directives,
     * session.gc_probability and session.gc_divisor
    */
    function gc($max) {
			$query="";
			
      if (!isset($this->gc_stmt)) {
        $query="DELETE FROM ".$this->dbDatabaseName." WHERE set_time < ?";
				$this->gc_stmt = $this->db->prepare();
      }
      
      $old = time() - $max;
      $this->gc_stmt->bind_param('s', $old);
      $this->gc_stmt->execute();
      
      return true;
    }
    
    /**
     * this function is used to get the unique key for encryption
     * from the sessions table
     * if there is no session it just returns a new random key for encryption
    */
    private function getkey($id) {
      $query="";
			
			if (!isset($this->key_stmt)) {
        $query="SELECT session_key FROM ".$this->dbDatabaseName." WHERE id = ? LIMIT 1";
				$this->key_stmt = $this->db->prepare($query);
      }
      
      $this->key_stmt->bind_param('s', $id);
      $this->key_stmt->execute();
      $this->key_stmt->store_result();
      if($this->key_stmt->num_rows == 1) { 
        $this->key_stmt->bind_result($key);
        $this->key_stmt->fetch();
        
        return $key;
      }
      else {
        $random_key = hash('sha512', uniqid(mt_rand(1, mt_getrandmax()), true));
        
        return $random_key;
      }
    }
    
    /**
     * this function generates a password salt as a string of x (default = 15) characters
     * in the a-zA-Z0-9!@#$%&*? range
     * @param $max integer The number of characters in the string
     * @return string
     */
    private function generateSalt($max = 15) {
      $characterList = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789!@#$%&*?";
      $i = 0;
      $salt = "";
      while ($i < $max) {
        $salt .= $characterList{mt_rand(0, (strlen($characterList) - 1))};
        $i++;
      }
      
      return $salt;
    }
    
    /**
     * these functions encrypt the data of the sessions,
     * they use a encryption key from the database which
     * is different for each session
     * we don't directly use that key in the encryption
     * but we use it to make the key hash even more random
    */
    private function encrypt($data, $key) {
      //$salt = 'cH!swe!retReGu7W6bEDRup7usuDUh9THeD2CHeGE*ewr4n39=E@rAsp7c-Ph@pH';
      $salt='=_|Y&$lDz0D07hL* l)QXHO+Z;xk|8>@Fw*h1ge*!d;7,~[7QqFs|WL#L@*V:O%l)';
      //$salt=$this->salt;
      $key = substr(hash('sha256', $salt.$key.$salt), 0, 32);
      $iv_size = mcrypt_get_iv_size(MCRYPT_RIJNDAEL_256, MCRYPT_MODE_ECB);
      $iv = mcrypt_create_iv($iv_size, MCRYPT_RAND);
      $encrypted = base64_encode(mcrypt_encrypt(MCRYPT_RIJNDAEL_256, $key, $data, MCRYPT_MODE_ECB, $iv));
      
      return $encrypted;
    }
    
    private function decrypt($data, $key) {
      //$salt = 'cH!swe!retReGu7W6bEDRup7usuDUh9THeD2CHeGE*ewr4n39=E@rAsp7c-Ph@pH';
      $salt='=_|Y&$lDz0D07hL* l)QXHO+Z;xk|8>@Fw*h1ge*!d;7,~[7QqFs|WL#L@*V:O%l)';
      //$salt=$this->salt;
      $key = substr(hash('sha256', $salt.$key.$salt), 0, 32);
      $iv_size = mcrypt_get_iv_size(MCRYPT_RIJNDAEL_256, MCRYPT_MODE_ECB);
      $iv = mcrypt_create_iv($iv_size, MCRYPT_RAND);
      $decrypted = mcrypt_decrypt(MCRYPT_RIJNDAEL_256, $key, base64_decode($data), MCRYPT_MODE_ECB, $iv);
      
      return $decrypted;
    }
  }
?>