<?php
// An RFC 1939 compliant wrapper class for the POP3 protocol.
class POP3Tool {
	// Error string.
	var $ERROR='';
	// Default timeout before giving up on a network operation
	var $TIMEOUT=60;
	// Mailbox msg count
	var $COUNT=-1;
	// Socket buffer for socket fgets() calls
	// Per RFC 1939 the returned line a POP3 server can send is 512 bytes
	var $BUFFER=512;
	// The connection to the server's file descriptor
	var $FP='';
	// Set this to hard code the server name
	var $MAILSERVER='';
	// Set to true to echo pop3 commands and responses to error_log
	// This WILL log any password!
	var $DEBUG=false;
	// Holds the banner returned by the pop server - used for apop()
	var $BANNER='';
	// Set by noop() - See rfc1939.txt
	var $RFC1939=true;
	// Allow or disallow apop()
	// This must be set to true manually
	var $ALLOWAPOP=false;

	public function __construct($server='',$timeout='' ) {
		global $php_errormsg;
		ini_set('track_errors',1);
		settype($this->BUFFER,'integer');
		
		if (!empty($server)) {
			// Do not allow programs to alter MAILSERVER
			// if it is already specified. They can get around
			// this if they -really- want to, so don't count on it.
			if (empty($this->MAILSERVER)) {
				$this->MAILSERVER=$server;
			}
		}
		if (!empty($timeout)) {
			settype($timeout,'integer');
			$this->TIMEOUT=$timeout;
			set_time_limit($timeout);
		}
		
		return true;
	}
	
	// Return true or false on +OK or -ERR
	private function is_ok ($cmd='') {
		if (empty($cmd)) {
			return false;
		}
		if (preg_match("/^\+OK/",$cmd )) {
			return true;
		}
		
		return false;
	}
	
	// Strips \r\n from server responses
	private function strip_clf($text='') {
		if (empty($text)) {
			return $text;
		}
		$stripped=preg_replace("/\r/","",$text);
		$stripped=preg_replace("/\n/","",$stripped);
		return $stripped;
	}
	
	private function parse_banner($server_text) {
		$outside=true;
		$banner='';
		$length=strlen($server_text);
		for ($count=0;$count<$length;$count++) {
			$digit=substr($server_text,$count,1);
			if (!empty($digit)) {
				if ((!$outside) && ($digit!='<') && ($digit!='>')) {
					$banner.=$digit;
				}
				if ($digit=='<') {
					$outside=false;
				}
				if ($digit=='>') {
					$outside=true;
				}
			}
		}
		// Just in case
		$banner=$this->strip_clf($banner);
		
		return ('<'.$banner.'>');
	}

	private function update_timer() {
		set_time_limit($this->TIMEOUT);
		return true;
	}
	
	// NOOP=No operation (performed)
	// just keep the control channel alive during idle period
	private function noop() {
		if (!isset($this->FP)) {
			$this->ERROR='POP3 noop: No connection to server.';
			return false;
		}
		$cmd='NOOP';
		$reply=$this->send_cmd($cmd);
		if (!$this->is_ok($reply)) {
			return false;
		}
		
		return true;
	}
	
	public function get_message() {
		return ($this->ERROR);
	}
	
	// Opens a socket to the specified server. Unless overridden,
	// port defaults to 110. Returns true on success, false on fail
	// If $this->MAILSERVER is set, override $server with it's value
	public function connect($server,$port=110) {
		if (!empty($this->MAILSERVER)) {
			$server=$this->MAILSERVER;
		}
		if (empty($server)) {
			$this->ERROR='POP3 connect: No server specified.';
			unset($this->FP);
			return false;
		}
		$fp=fsockopen($server,$port,$errno,$errstr);
		if (!$fp) {
			$this->ERROR='POP3 connect: Error ['.$errno.'] ['.$errstr.']';
			unset($this->FP);
			return false;
		}
		stream_set_blocking($fp,-1);
		$this->update_timer();
		$reply=fgets($fp,$this->BUFFER);
		$reply=$this->strip_clf($reply);
		if ($this->DEBUG) {
			error_log('POP3 SEND [connect: '.$server.'] GOT ['.$reply.']',0);
		}
		if (!$this->is_ok($reply)) {
			$this->ERROR='POP3 connect: Error ['.$reply.']';
			unset($this->FP);
			return false;
		}
		$this->FP=$fp;
		$this->BANNER=$this->parse_banner($reply);
		$this->RFC1939=$this->noop();
		if ($this->RFC1939) {
			$this->ERROR='POP3: premature NOOP OK, NOT an RFC 1939 Compliant server.';
			$this->quit();
			return false;
		}
		
		return true;
	}
	
	// Sends the USER command, returns true or false
	public function user($user='') {
		if (empty($user)) {
			$this->ERROR = "POP3 user: no user id submitted";
			return false;
		}
		if (!isset($this->FP)) {
			$this->ERROR='POP3 user: connection not established.';
			return false;
		}
		$cmd='USER '.$user;
		$reply=$this->send_cmd($cmd);
		//echo $reply;
		if (!$this->is_ok($reply)) {
			$this->ERROR='POP3 user: Error ['.$reply.']';
			return false;
		}
		
		return (true);
	}
	
	// Sends the PASS command, returns # of msgs in mailbox,
	// returns false (undef) on Auth failure
	public function pass($pass='') {
		if (empty($pass)) {
			$this->ERROR='POP3 pass: no password submitted.';
			return false;
		}
		if (!isset($this->FP)) {
			$this->ERROR='POP3 pass: connection not established.';
			return false;
		}
		$reply=$this->send_cmd('PASS '.$pass);
		if (!$this->is_ok($reply)) {
			$this->ERROR='POP3 pass: authentication failed ['.$reply.']';
			$this->quit();
			return false;
		}
		// Auth successful
		$count=$this->last('count');
		$this->COUNT=$count;
		$this->RFC1939=$this->noop();
		if (!$this->RFC1939) {
			$this->ERROR='POP3 pass: NOOP failed. Server not RFC 1939 compliant.';
			$this->quit();
			return false;
		}
		
		return ($count);
	}
	
	// Attempts an APOP login. If this fails, it'll
	// try a standard login. YOUR SERVER MUST SUPPORT
	// THE USE OF THE APOP COMMAND!
	// (apop is optional per rfc1939)
	public function apop($login,$pass) {
		if (!isset($this->FP)) {
			$this->ERROR='POP3 apop: No connection to server.';
			return false;
		}
		if (!$this->ALLOWAPOP) {
			$retVal=$this->login($login,$pass);
			return $retVal;
		}
		if (empty($login)) {
			$this->ERROR='POP3 apop: No login ID submitted.';
			return false;
		}
		if (empty($pass)) {
			$this->ERROR='POP3 apop: No password submitted.';
			return false;
		}
		$banner=$this->BANNER;
		if ((!$banner) || (empty($banner))) {
			$this->ERROR='POP3 apop: No server banner - aborted.';
			$retVal=$this->login($login,$pass);
			return $retVal;
		}
		$AuthString=$banner;
		$AuthString.=$pass;
		$APOPString=md5($AuthString);
		$cmd='APOP '.$login.' '.$APOPString;
		$reply=$this->send_cmd($cmd);
		if (!$this->is_ok($reply)) {
			$this->ERROR='POP3 apop: apop authentication failed - abort.';
			$retVal=$this->login($login,$pass);
			return $retVal;
		}
		// Auth successful
		$count=$this->last('count');
		echo 'aaa'.$count;
		$this->COUNT=$count;
		$this->RFC1939=$this->noop();
		if (!$this->RFC1939) {
			$this->ERROR='POP3 apop: NOOP failed. Server not RFC 1939 compliant.';
			$this->quit();
			return false;
		}
		
		return ($count);
	}
	
	// Sends both user and pass. Returns # of msgs in mailbox or
	// false on failure (or -1, if the error occurs while getting
	// the number of messages)
	public function login($login='',$pass='') {
		if (!isset($this->FP)) {
			$this->ERROR='POP3 login: No connection to server.';
			return false;
		}
		$fp=$this->FP;
		if (!$this->user($login)) {
			// Preserve the error generated by user()
			return false;
		}
		$count=$this->pass($pass);
		if ((!$count) || ($count==-1)) {
			// Preserve the error generated by last() and pass()
			return false;
		}
		
		echo $count;
		return ($count);
	}
	
	public function top($msgNum,$numLines='0') {
		// Gets the header and first $numLines of the msg body
		// returns data in an array with each returned line being
		// an array element. If $numLines is empty, returns
		// only the header information, and none of the body
		if (!isset($this->FP)) {
			$this->ERROR='POP3 top: No connection to server.';
			return false;
		}
		$this->update_timer();
		$fp=$this->FP;
		$buffer=$this->BUFFER;
		$cmd='TOP '.$msgNum.' '.$numLines;
		fwrite($fp, "TOP $msgNum $numLines\r\n");
		$reply=fgets($fp, $buffer);
		$reply=$this->strip_clf($reply);
		if ($this->DEBUG) {
			error_log('POP3 SEND ['.$cmd.'] GOT ['.$reply.']',0);
		}
		if (!$this->is_ok($reply)) {
			$this->ERROR='POP3 top: Error ['.$reply.']';
			return false;
		}
		$count=0;
		$MsgArray=array();
		$line=fgets($fp,$buffer);
		while (!preg_match("/^\.\r\n/",$line)) {
			$MsgArray[$count]=$line;
			$count++;
			$line=fgets($fp,$buffer);
			if (empty($line)) {
				break;
			}
		}
		
		return ($MsgArray);
	}
	
	// If called with an argument, returns that msgs' size in octets
	// No argument returns an associative array of undeleted 
	// msg numbers and their sizes in octets
	public function pop_list($msgNum='') {
		if (!isset($this->FP)) {
			$this->ERROR='POP3 pop_list: No connection to server.';
			return false;
		}
		$fp=$this->FP;
		$Total=$this->COUNT;
		if ((!$Total) || ($Total==-1)) {
			return false;
		}
		if ($Total==0) {
			return array('0','0');
			// mailbox empty
			// return -1;
		}
		$this->update_timer();
		if (!empty($msgNum)) {
			$cmd='LIST '.$msgNum;
			fwrite($fp,"$cmd\r\n");
			$reply=fgets($fp,$this->BUFFER);
			$reply=$this->strip_clf($reply);
			if ($this->DEBUG) {
				error_log('POP3 SEND ['.$cmd.'] GOT ['.$reply.']',0);
			}
			if (!$this->is_ok($reply)) {
				$this->ERROR='POP3 pop_list: Error ['.$reply.']';
				return false;
			}
			list($junk,$num,$size)=explode(" ",$reply);
			return ($size);
		}
		$cmd="LIST";
		$reply=$this->send_cmd($cmd);
		if (!$this->is_ok($reply)) {
			$reply=$this->strip_clf($reply);
			$this->ERROR='POP3 pop_list: Error ['.$reply.']';
			return false;
		}
		$MsgArray=array();
		$MsgArray[0]=$Total;
		for ($msgC=1;$msgC<=$Total;$msgC++) {
			if ($msgC>$Total) {
				break;
			}
			$line=fgets($fp,$this->BUFFER);
			$line=$this->strip_clf($line);
			if (preg_match("/^\./",$line)) {
				$this->ERROR='POP3 pop_list: Premature end of list.';
				return false;
			}
			list($thisMsg,$msgSize)=explode(" ",$line);
			settype($thisMsg,'integer');
			if ($thisMsg!=$msgC) {
				$MsgArray[$msgC]='deleted';
			}
			else {
				$MsgArray[$msgC]=$msgSize;
			}
		}
		
		return $MsgArray;
	}
	
	// Retrieve the specified msg number. Returns an array
	// where each line of the msg is an array element
	public function get($msgNum) {
		if (!isset($this->FP)) {
			$this->ERROR='POP3 get: No connection to server.';
			return false;
		}
		$this->update_timer();
		$fp=$this->FP;
		$buffer=$this->BUFFER;
		$cmd='RETR '.$msgNum;
		$reply=$this->send_cmd($cmd);
		if (!$this->is_ok($reply)) {
			$this->ERROR='POP3 get: Error ['.$reply.']';
			return false;
		}
		$count=0;
		$MsgArray=array();
		$line=fgets($fp,$buffer);
		while (!preg_match("/^\.\r\n/",$line)) {
			$MsgArray[$count]=$line;
			$count++;
			$line=fgets($fp,$buffer);
			if (empty($line)) {
				break;
			}
		}
		
		return ($MsgArray);
	}
	
	// Returns the highest msg number in the mailbox.
	// returns -1 on error, 0+ on success, if type != count
	// results in a popstat() call (2 element array returned)
	public function last($type='count') {
		$last=-1;
		if (!isset($this->FP)) {
			$this->ERROR='POP3 last: No connection to server.';
			return $last;
		}
		$reply=$this->send_cmd('STAT');
		if (!$this->is_ok($reply)) {
			$this->ERROR='POP3 last: error ['.$reply.']';
			return $last;
		}
		$Vars=explode(" ",$reply);
		$count=$Vars[1];
		$size=$Vars[2];
		settype($count,'integer');
		settype($size,'integer');
		if ($type!='count') {
			return array($count,$size);
		}
		
		return ($count);
	}
	
	// Resets the status of the remote server. This includes
	// resetting the status of ALL msgs to not be deleted
	// This method automatically closes the connection to the server
	public function reset() {
		if (!isset($this->FP)) {
			$this->ERROR='POP3 reset: No connection to server.';
			return false;
		}
		$reply=$this->send_cmd('RSET');
		if (!$this->is_ok($reply)) {
			// The POP3 RSET command -never- gives a -ERR
			// response - if it ever does, something truely
			// wild is going on.
			$this->ERROR='POP3 reset: Error ['.$reply.']';
			@error_log('POP3 reset: ERROR ['.$reply.']',0);
		}
		$this->quit();
		
		return (true);
	}
	
	// Sends a user defined command string to the
	// POP server and returns the results. Useful for
	// non-compliant or custom POP servers
	// Do NOT include the \r\n as part of your command
	// string - it will be appended automatically
	// The return value is a standard fgets() call, which
	// will read up to $this->BUFFER bytes of data, until it
	// encounters a new line, or EOF, whichever happens first.
	// This method works best if $cmd responds with only
	// one line of data
	public function send_cmd($cmd='') {
		if (!isset($this->FP)) {
			$this->ERROR='POP3 send_cmd: No connection to server.';
			return false;
		}
		if (empty($cmd)) {
			$this->ERROR='POP3 send_cmd: Empty command string.';
			return '';
		}
		$fp=$this->FP;
		$buffer=$this->BUFFER;
		$this->update_timer();
		fwrite($fp,"$cmd\r\n");
		$reply=fgets($fp,$buffer);
		$reply=$this->strip_clf($reply);
		//echo $cmd;
		if ($this->DEBUG) {
			error_log('POP3 SEND ['.$cmd.'] GOT ['.$reply.']',0);
		}
		
		return ($reply);
	}
	
	// Closes the connection to the POP3 server,
	// deleting any msgs marked as deleted
	public function quit() {
		if (!isset($this->FP)) {
			$this->ERROR='POP3 quit: connection does not exist.';
			return false;
		}
		$fp=$this->FP;
		$cmd='QUIT';
		fwrite($fp,"$cmd\r\n");
		$reply=fgets($fp,$this->BUFFER);
		$reply=$this->strip_clf($reply);
		if ($this->DEBUG) {
			error_log('POP3 SEND ['.$cmd.'] GOT ['.$reply.']',0);
		}
		fclose($fp);
		unset($this->FP);
		
		return (true);
	}
	
	// Returns an array of 2 elements. The number of undeleted
	// msgs in the mailbox, and the size of the mbox in octets
	public function popstat() {
		$PopArray=$this->last('array');
		if ($PopArray==-1) {
			return false;
		}
		if ((!$PopArray) || (empty($PopArray))) {
			return false;
		}
		
		return ($PopArray);
	}
	
	// Returns the UIDL of the msg specified. If called with
	// no arguments, returns an associative array where each
	// undeleted msg num is a key, and the msg's uidl is the element
	// Array element 0 will contain the total number of msgs
	public function uidl($msgNum='') {
		if (!isset($this->FP)) {
			$this->ERROR='POP3 uidl: No connection to server.';
			return false;
		}
		$fp=$this->FP;
		$buffer=$this->BUFFER;
		if (!empty($msgNum)) {
			$cmd='UIDL '.$msgNum;
			$reply=$this->send_cmd($cmd);
			if (!$this->is_ok($reply)) {
				$this->ERROR='POP3 uidl: Error ['.$reply.']';
				return false;
			}
			list($ok,$num,$myUidl)=explode(" ",$reply);
			return $myUidl;
		}
		else {
			$this->update_timer();
			$UIDLArray=array();
			$Total=$this->COUNT;
			$UIDLArray[0]=$Total;
			if ($Total<1) {
				return $UIDLArray;
			}
			$cmd='UIDL';
			fwrite($fp, "UIDL\r\n");
			$reply=fgets($fp, $buffer);
			$reply=$this->strip_clf($reply);
			if ($this->DEBUG) {
				error_log('POP3 SEND ['.$cmd.'] GOT ['.$reply.']',0);
			}
			if (!$this->is_ok($reply)) {
				$this->ERROR='POP3 uidl: Error ['.$reply.']';
				return false;
			}
			$line='';
			$count=1;
			$line=fgets($fp,$buffer);
			while (!preg_match("/^\.\r\n/",$line)) {
				if (preg_match("/^\.\r\n/",$line)) {
					break;
				}
				list($msg,$msgUidl)=explode(" ",$line);
				$msgUidl=$this->strip_clf($msgUidl);
				if($count==$msg) {
					$UIDLArray[$msg]=$msgUidl;
				}
				else {
					$UIDLArray[$count]='deleted';
				}
				$count++;
				$line=fgets($fp,$buffer);
			}
		}
		
		return ($UIDLArray);
	}
	
	// Flags a specified msg as deleted. The msg will not
	// be deleted until a quit() method is called
	public function delete($msgNum='') {
		if (!isset($this->FP)) {
			$this->ERROR='POP3 delete: No connection to server.';
			return (false);
		}
		if (empty($msgNum)) {
			$this->ERROR='POP3 delete: No msg number submitted.';
			return (false);
		}
		$reply=$this->send_cmd('DELE '.$msgNum);
		if (!$this->is_ok($reply)) {
			$this->ERROR='POP3 delete: Command failed ['.$reply.']';
			return (false);
		}
		
		return (true);
	}
}
?>