<?php
  namespace aniDateTime;
  
  /**
   * A utility class to help validate and convert Date argument values.
  **/
  class Preconditions {
    /**
     * A private constructor. Use the static factory methods to create DateTime
     * instances.
    **/
    private function __construct() { }
    
    /**
     * Ensure an argument is a valid time zone. If $value is not of type
     * Bronto\Date\TimeZone or a string representation of a time zone, throw an
     * InvalidArgumentException.
     *
     * @param Bronto\Date\TimeZone|string $value a TimeZone object, or a string
     * representation of a time zone
     * @param string $argumentName the name of the argument, to be used if an
     * exception must be thrown
     *
     * @throws \InvalidArgumentException if $value is not a TimeZone or a valid
     * string representation of a time zone
     *
     * @return a Bronto\Date\TimeZone object
    **/
    public static function requireTimeZone($timeZone, $argumentName) {
      if ($timeZone instanceof TimeZone) {
        return $timeZone;
      }

      if (!is_string($timeZone)) {
        throw new \InvalidArgumentException("$argumentName must a Bronto\\Date\\TimeZone object or a time zone string [\$timeZone = " . print_r($timeZone, true));
      }

      return TimeZone::parse($timeZone);
    }
    
    /**
     * Ensure an argument is of type int. If $value is not of type int or cannot
     * trivially be interpretted as an int, throw an InvalidArgumentException.
     *
     * @param int|string $value an int value, or a string representation of an
     * int
     * @param string $argumentName the name of the argument, to be used if an
     * exception must be thrown
     *
     * @throws \InvalidArgumentException if $value is not an int or a string
     * representation of an int
     *
     * @return the int value
    **/
    public static function requireInt($value, $argumentName) {
      if (is_int($value))
        return $value;
      
      if (!is_string($value) || !is_numeric($value))
        throw new \InvalidArgumentException("$argumentName must be an integer");
      
      if ($value === '0') {
        return 0;
      }
      
      $intValue = intval($value);
      // This checks that $value really represents an int, and not a
      // float/double.
      if (((string) $intValue) !== $value)
        throw new \InvalidArgumentException("$argumentName must be an integer");
      
      return $intValue;
    }
    
    /**
     * Ensure an argument is of type float. If $value is not of type float or cannot
     * trivially be interpretted as a float, throw an InvalidArgumentException.
     *
     * @param float|string $value an float value, or a string representation of a
     * float
     * @param string $argumentName the name of the argument, to be used if an
     * exception must be thrown
     *
     * @throws \InvalidArgumentException if $value is not a float or a string
     * representation of a float
     *
     * @return the float value
    **/
    public static function requireFloat($value, $argumentName) {
      if (is_float($value))
        return $value;
      
      echo $value;
      
      if (!is_string($value) || !is_numeric($value))
        throw new \InvalidArgumentException("$argumentName must be a float");
      
      if ($value === '0') {
        return 0;
      }
      
      $floatValue = floatval($value);
      // This checks that $value really represents an int, and not a
      // float/double.
      if (((string) $floatValue) !== $value)
        throw new \InvalidArgumentException("$argumentName must be a float");
      
      return $floatValue;
    }
    

    /**
     * Ensure an argument is of type string. If $value is not a string, throw an
     * InvalidArgumentException.
     *
     * @param string $value a string value
     * @param string $argumentName the name of the argument, to be used if an
     * exception must be thrown
     *
     * @throws \InvalidArgumentException if $value is not a string
     *
     * @return the string value
     */
    public static function requireString($value, $argumentName) {
      if (!is_string($value)) {
        throw new \InvalidArgumentException("$argumentName must be a string");
      }

      return $value;
    }
    
    /**
     * Ensure an argument is non-null.
     *
     * @param mixed $argument an argument
     * @param string $argumentName the name of the argument, to be used if an
     * exception must be thrown
     *
     * @throws \InvalidArgumentException if $argument is null
     */
    public static function checkNotNull($argument, $argumentName) {
      if (is_null($argument)) {
        throw new \InvalidArgumentException("$argumentName cannot be null");
      }
    }
  }
?>