<?php
  class CacheTool {
    /* must be changed before using */
    private $_salt="change_the_password";
    private bool $isSalt=false;
    private string $_name="";
    private string $_dir="";
    private string $_extension="";
    private string $_path="";
    private bool $_autoSave=false;
    private array $_cache=array();
    private bool $useAPCU=false;
    
    /* constructor: cache name, path, extension, isSalt: whether or not use the '$_salt' string */
    public function __construct($useAPCU,string $name="aniCache",string $dir="",string $extension=".cache",bool $isSalt=false) {
      if (gettype($useAPCU)!="boolean")
        die("It cannot be decided whether or not you want to use the APCU extension! Please modify the first parameter of the class to a boolean type!");
      
      if (!isset($useAPCU))
        $this->useAPCU=false;
      else
        $this->useAPCU=$useAPCU;
      
      if (!$this->useAPCU) {
        if (!isset($isSalt))
          $this->isSalt=false;
        else
          $this->isSalt=$isSalt;
        
        if ($this->isSalt) {
          if (md5($this->_salt) == "c3163b781ac4f49f0af492860f2ef1f8") {
            throw new Exception("CacheTool attention: change _salt value before usage! (line 4)");
            die();
          }
        }
        
        $dir=str_replace("\\", "/", $dir);
        if (!$this->endsWith($dir,"/"))
          $dir.="/";
        
        $this->_name=$name;
        if (($dir=="") || ($dir=="/"))
          die("If the APCU usage is false then the cache direcory cannot be empty! Usage: useAPCU(false | true),cacheName,cacheDir,cacheExtension,useSalt(false | true).");
        else
          $this->_dir=$dir;
        $this->_extension=$extension;
        $this->_path=$this->getCachePath();
        $this->checkCacheDir();
        $this->loadCache();
      }
    }
    
    /*******************/
    /* Private methods */
    /*******************/
    
    /* check a given cache object whether or not has been expired */
    private function isExpired($data) : bool {
      if ($data["e"] == -1)
        return false;
      
      $expiresOn=$data["t"]+$data["e"];
      return $expiresOn < time();
    }
    
    /* get the cache path */
    private function getCachePath() : string {
      if ($this->isSalt)
        return $this->_dir.md5($this->_name.$this->_salt).$this->_extension;
      else
        return $this->_dir.$this->_getHash($this->_path).$this->_extension;
    }
    
    /* check the given cache directory properties: can be created or is writable */
    private function checkCacheDir() : bool {
      if (!is_dir($this->_dir) && !mkdir($this->_dir,0775,true))
        throw new Exception("Unable to create cache directory ($this->_dir)");
      
      if (!is_writable($this->_dir) || !is_readable($this->_dir)) {
        if (!chmod($this->_dir,0775))
          throw new Exception("Cache directory must be readable and writable ($this->_dir)");
      }
      
      return true;
    }
    
    /* check a string: whether or not ends with the given string */
    private function endsWith(string $data,string $endsWith) : bool {
        $length=strlen($endsWith);
        return $length === 0 || (substr($data, -$length) === $endsWith);
    }
    
    /* load the cache content from the given '$this->_path' */
    private function loadCache() : bool {
      if (!file_exists($this->_path))
        return false;
      
      $content=file_get_contents($this->_path);
      $this->_cache=json_decode($content,true);
      
      return true;
    }
    
    /* get the hash value of a string */
    private function _getHash($filename) {
      return sha1($filename);
    }
    
    /************************/
    /* END: Private methods */
    /************************/
    
    /******************/
    /* Public methods */
    /******************/
    
    /* set the autosave function */
    public function setAutoSave(bool $state) : void {
      if (!$this->useAPCU)
        $this->_autoSave=$state;
    }
    
    /* get one of the key:value pair of the current cache object content */
    public function Getter(string $key) : string {
      $retval="";
      
      if (!$this->useAPCU) {
        if ($this->_cache === null)
          return false;
        if (!array_key_exists($key,$this->_cache))
          return false;
        
        $data=$this->_cache[$key];
        if ($this->isExpired($data)) {
            unset($this->_cache[$key]);
            
            if ($this->_autoSave)
                $this->saveCache();
            
            return false;
        }
        
        $retval=unserialize($data["v"]);
      }
      
      return $retval;
    }
    
    /* set a key:value pair of the current cache object content (new or old one) */
    public function Setter(string $key,$value,int $ttl=-1) : void {
      if (!$this->useAPCU) {
        $data=[
            "t" => time(),
            "e" => $ttl,
            "v" => serialize($value),
        ];
        
        if ($this->_cache === null) {
          $this->_cache = [
              $key => $data,
          ];
        }
        else {
          $this->_cache[$key]=$data;
        }
        
        if ($this->_autoSave)
          $this->saveCache();
      }
    }
    
    /* delete a key:value pair of the current cache object content */
    public function Delete(string $key) : bool {
      if (!$this->useAPCU) {
        if ($this->_cache === null)
          return false;
        if (!array_key_exists($key, $this->_cache))
          return false;
        
        unset($this->_cache[$key]);
        
        if ($this->_autoSave)
          $this->saveCache();
        
        return true;
      }
      
      return false;
    }
    
    /* delete expired cache content */
    public function deleteExpired() : bool {
      if (!$this->useAPCU) {
        if ($this->_cache === null)
          return false;
        
        foreach ($this->_cache as $key => $value) {
          if ($this->isExpired($value))
            unset($this->_cache[$key]);
        }
        
        if ($this->_autoSave)
          $this->saveCache();
        
        return true;
      }
      
      return false;
    }
    
    /* manually save the cache content into the given '$this->_path' */
    public function saveCache() : bool {
      if (!$this->useAPCU) {
        if ($this->_cache === null)
          return false;
        
        $content=json_encode($this->_cache);
        file_put_contents($this->_path,$content);
        return true;
      }
      
      return false;
    }
    
    /********************/
    /* APCU extension   */
    /* used only memory */
    /********************/
    
    /* check APCU extension */
    public function _checkAPCU() : bool {
      if ($this->useAPCU) {
        if ((extension_loaded('apcu')) && (apcu_enabled() || ini_get('apc.enabled')))
          return true;
        else
          return false;
      }
      
      return false;
    }
    
    /* add to the APCU */
    public function _addAPCU($key,$value,$ttl=0) : bool {
      if ($this->useAPCU) {
        if (!apcu_exists($key))
          return apcu_add($key,$value,$ttl);
        else
          return false;
      }
      
      return false;
    }
    
    /* delete from the APCU */
    public function _delAPCU($key) {
      if ($this->useAPCU) {
        if (apcu_exists($key))
          apcu_delete($key);
      }
    }
    
    /* update a value of the APCU cache */
    public function _updateAPCU($key,$value,$ttl=0) : bool {
      if ($this->useAPCU)
        return apcu_store($key,$value,$ttl);
      
      return false;
    }
    
    /* get a value from the APCU cache */
    public function _fetchAPCU($key) {
      if ($this->useAPCU) {
        if (apcu_exists($key))
          return apcu_fetch($key);
        else
          return null;
      }
    }
    
    /* clear the APCU cache content */
    public function _clearAPCU() {
      if ($this->useAPCU)
        apcu_clear_cache();
    }
    
    /* get the APCU  content */
    public function _infoAPCU() {
      if ($this->useAPCU)
        return apcu_cache_info();
    }
    
    /* retrieve the APCU Shared Memory Allocation information */
    public function _smaInfoAPCU() {
      if ($this->useAPCU)
        return apcu_sma_info();
    }
    
    /*************************/
    /* END: APCU extension   */
    /* used only memory      */
    /*************************/
    
    /***********************/
    /* END: Public methods */
    /***********************/
  }
?>
