<?php
  /*
  * error handling class
  * This class encapsulates PHP's error handling functions. Once instantated it will become the default error handler
  * in your PHP script. Each error will be written to a log file, with the option of having certain errors sent to an
  * adminsitrator
  */
   
  class SessionErrorHandlerTool {
    // PUBLIC PROPERTIES
    /* path of the error log file */
    var $logpath;
    /* 
    * turn this flag on to send error messages to the browser
    * as well as the log file: useful while developing
    */
    var $debug_output = false;
    /*
    * an array of error constants which are considered fatal
    * if these errors oare encountered the script will terminate
    */   
    var $fatal_errors = Array (E_USER_ERROR);
    /*
    * fatal error function called before the script is terminated
    * this allows you to display a friendly error message to your users
    *
    * The property must be an array. The first element is the function name
    * the remaining elements are the function parameters.
    */
    var $fatal_error_function;
    /* 
    * array of error constants pertaining to errors which will be sent 
    * via email to the administrator
    */
    var $email_errors = Array (E_USER_ERROR, E_WARNING, E_NOTICE);
    /* administrators email address */
    var $admin_email;
    /* subject of the email sent to admin */    
    var $email_subject = 'PHP Script Errors';
    /* miniumum number of minutes between sending errors - this helps prevent DOS attacks */
    var $email_send = 5;
    /* path of the temporary file to store administrator email messages */
    var $email_tmppath = 'emailerror.tmp';
    /* set to true to turn on email support */    
    var $email_on_errors = false;
    
    /*
    * class constructor - sets the error handler
    * please see instructions on instantating this class
    */
    public function __construct($currLogPath='error.log') {
      $this->logpath=$currLogPath;
      /* set the error handler */
      set_error_handler(Array (&$this, 'ErrCallback'));
    }    
    
    /*
    * error call back function - this will be invoked every time the
    * a non fatal PHP error occurs and a fatal user error occurs
    */     
    public function ErrCallback($errno, $errmsg, $filename, $line, $vardump) {
      $types = Array (E_ERROR => 'FATAL',
                      E_WARNING => 'WARNING',
                      E_NOTICE => 'NOTICE',
                      E_USER_ERROR => 'USER FATAL',
                      E_USER_WARNING => 'USER WARNING',
                      E_USER_NOTICE => 'USER NOTICE');
      $time = Date('r');
      /* send an error to the broswer if the user has turned on the debug_output flag */
      if ($this->debug_output) {
        $debugmsg = "<b>{$types[$errno]}:</b> $errmsg in <b>$filename</b> on line $line<br />\n";
        echo ($debugmsg);
      }
      
      /* if email support is turned on and the error is in the email_errors array */
      if ($this->email_on_errors && in_array($errno, $this->email_errors)) {
        if (! $this->admin_email) {
          /* turn off email errors and send a warning to the user */
          $this->email_on_errors = false;
          trigger_error('CLASS error->err_callback: No administrator email set. Turning off email support',
                        E_USER_WARNING);
        }
        else {
          /* generate an error string */
          $vars = print_r ($vardump, true);
          $errmsg = "--------------------------------------------------------------------\n" .
                    "Time: $time\nError Type: {$types[$errno]}\nFile: $filename\n" .
                    "Message: $errmsg\nLine: $line\n\nVariables:\n$vars\n" .
                    "--------------------------------------------------------------------\n";
          $sendtime = 60 * $this->email_send;
          
          if (file_exists($this->email_tmppath) && (time() - filectime($this->email_tmppath) > $sendtime)) {
            /* send an email to the administrator */
            $errmsg = file_get_contents($this->email_tmppath) . $errmsg;
            if ($errmsg === false) {
              trigger_error('CLASS error->err_callback: Error retrieving contents of temporary file.',
                            E_USER_WARNING);
            }
            if (! mail($this->admin_email, $this->email_subject, $errmsg)) {
              trigger_error('CLASS error->err_callback: Error sending administrator email.', E_USER_WARNING);
            }
            else {
              /* 
              * delete the temporary error file once sent
              * N.b: Faliure to remove this file will result in a fatal error which kills the script. This
              * prevents the scritp sending multiple e-mails
              */
              if (! unlink($this->email_tmppath)) {
                trigger_error('CLASS error->err_callback: Error removing temporary log file.', E_USER_ERROR);
              }
            }
          }
          else {
            /* append the error string to a temporary error file */
            $fhwnd = fopen($this->email_tmppath, 'a');
            if (! $fhwnd) {
            trigger_error('CLASS error->err_callback: Error opening temporary log file for writing.',
                          E_USER_WARNING);
            }
            fwrite ($fhwnd, $errmsg);
            fclose ($fhwnd);
          }
        }
      }
      /* create a line for the error file */
      $errline = "$time\t{$types[$errno]}\t$filename\t$line\t$errmsg\n";
      
      /* append it to the log file */
      if (! error_log ($errline, 3, $this->logpath)) {
        trigger_error('CLASS error->err_callback: Error writing to logfile', E_USER_ERROR);
      }
      
      /* if this is a fatal error kill the script */
      if (in_array($errno, $this->fatal_errors)) {
        if ($this->fatal_error_function) {
          if (is_array($this->fatal_error_function) && (count($this->fatal_error_function >= 1))) {
            call_user_func_array ($this->fatal_error_function[0], array_slice($this->fatal_error_function, 1));
          }
          else {
            trigger_error('CLASS error->err_callback: fatal_error_function property in wrong format.' .
                          'skipping function call.', E_USER_WARNING);
          }
        }
        die ();
      }
    }
  }
?>