<?php
  # PHP Data Object (PDO) Tool Class
  # can be used with:
  # MySQL/MariaDB, PostgreSQL, Sqlite, Microsoft SQL
  # Intebase, IBM DB2 and Oracle RDBMS
  #
	# created by garai, June 2014.
  
	require_once '__pdo_tool_manifest.php';
	
  class PDOTool extends PDO {
    /*****************
     * ALL VARIABLES *
    ******************/
		
    /* the current PDO object */
    private static $oPDO=null;
		/* PHP Statement Handler object */
    private $_oSTH=null;
    /* data source name for the PDO */
    private $strDsn='';
    /* database username */
    private $strUserName='';
    /* database password for the username */
    private $strPasswd='';
    /* connection mode */
    private $strMode=null;
    /* array to get query parameters */
    private $aData=null;
    /**
     * error mode in PDO
     * 
     * @var PDO::ERRMODE_SILENT
     * @var PDO::ERRMODE_WARNING
     * @var PDO::ERRMODE_EXCEPTION
    **/
    private $errorMode=null;
    /* array to debug the SQL statement: PDO error message */
    private $strError=null;
    /* set print_r command instead of echo, print */
    private $errorCallbackFunction;
		/* error output format */
    private $errorMsgFormat;
		/* enable/disable class debug mode boolean */
    private $log=false;
    /* set flag for batch insert boolean */
    private $batch=false;
		/* PDO SQL Statement */
    private $sSql='';
		/* get all PDO affected rows */
    private $iAffectedRows=0;
		/* set PDO valid Query operation array */
    private $aValidOperation=array('SELECT','INSERT','UPDATE','DELETE');
		/* PDO Error message array */
    private static $error_messages=array('You have not passed any valid database config array key!',
                                         'You have not set any valid database config array key!',
                                         'You have not set any valid database config array!',
                                         'Invalid operation called in query. Use only ',
                                         'The query string is empty!',
                                         'The table name has not been found!',
                                         'The data is not in valid format!',
                                         'The update statement is not in valid format!',
                                         'It is not a valid WHERE condition!',
                                         'Error message from the PDOTool class: ',
                                         'The driver you have added is not supported by the server, now it will quit!');
		/* Database character sets */
		private $charset=array(
													 'big5',
													 'dec8',
													 'cp850',
													 'hp8',
													 'koi8r',
													 'latin1',
													 'latin2',
													 'swe7',
													 'ascii',
													 'ujis',
													 'sjis',
													 'hebrew',
													 'tis620',
													 'euckr',
													 'koi8u',
													 'gb2312',
													 'greek',
													 'cp1250',
													 'gbk',
													 'latin5',
													 'armscii8',
													 'utf8',
													 'ucs2',
													 'cp866',
													 'keybcs2',
													 'macce',
													 'macroman',
													 'cp852',
													 'latin7',
													 'utf8mb4',
													 'cp1251',
													 'utf16',
													 'cp1256',
													 'cp1257',
													 'utf32',
													 'binary',
													 'geostd8',
													 'cp932',
													 'eucjpms'
													);
    
		/*****************
		 * CLASS METHODS *
		******************/
    
		/**
     * Constructor
    **/
    public function __construct($dsn,$uname,$pwd,$mode=false,$errMode=PDO::ERRMODE_EXCEPTION,$codepage='UTF8') {
      $this->strDsn=$dsn;
      $this->strUserName=$uname;
      $this->strPasswd=$pwd;
      $this->strMode=$mode;
      $this->errorMode=$errMode;
			$driver='';
			$options='';
      
      $options=array(PDO::ATTR_PERSISTENT => $this->strMode, 
                     PDO::ATTR_ERRMODE => $this->errorMode
			);
			
			/* connect to the RDBMS */
      try {
        /* check the given driver */
        $driver=$this->readDriver($this->strDsn);
				if (!$this->GetSupportedDrivers($driver))
					die(self::$error_messages[9].self::$error_messages[10].'<br />'.$this->strDsn);
				/* connection method */
				if ($driver=='mysql') {
					foreach ($this->charset as $index => $value) {
						if ($codepage==$value) {
							/* get MySQL/MariaDB character set */
							$options = array(PDO::MYSQL_ATTR_INIT_COMMAND => 'SET NAMES '.$codepage);
							break;
						}
					}
					parent::__construct($this->strDsn,$this->strUserName,$this->strPasswd,$options);
				}
				else {
					parent::__construct($this->strDsn,$this->strUserName,$this->strPasswd);
				}

        
        /* set PDO error mode */
        $this->setAttribute(PDO::ATTR_ERRMODE,$this->errorMode);
        
        /* use this setting to force PDO to either always emulate prepared statements (if TRUE),
        or to try to use native prepared statements (if FALSE) */
        $this->setAttribute(PDO::ATTR_EMULATE_PREPARES,true);
        
        /* set default PDO fetch mode as fetch assoc */
        $this->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE,PDO::FETCH_ASSOC);
        
        /* set the case in which to return column_names */
        $this->setAttribute(PDO::ATTR_CASE,PDO::CASE_NATURAL);
      }
      catch (PDOException $e) {
				die(self::$error_messages[9].$e->getMessage());
      }
    }
    
    /**
     * Destructor, unset the class object PDO
    **/
    public function __destruct() {
      self::$oPDO = null;
    }
    
    /*******************************/
    /******* PRIVATE METHODS *******/
    /*******************************/
		
		/**
		 * Read the driver type from the given dsn string
		 *
		 * @param boolean
		**/ 
		private function readDriver($dsn) {
			$result='';
			$i=0;
			
			while ($dsn[$i]!==':') {
				$result.=$dsn[$i];
				++$i;
			}
			
			return ($result);
		}
		
		/**
		 * GetSupportedDrivers - return with the available drivers of the server
		 *
		 * @param string with the supported RDBMS list
		 * @param boolean with the result whether supported the given driver or not
		**/
		private function GetSupportedDrivers($currDriver='') {
			/* all supported PDO drivers of the server */
			$pdoDriver=array();
			/* is supported driver */
			$result=false;
			
			foreach(PDO::getAvailableDrivers() as $driver) {
			  $pdoDriver[]=$driver;
			}
			
			if ($currDriver==='')
				return ($pdoDriver);
			else {
				foreach ($pdoDriver as $key => $value) {
					if (trim($value)===trim($currDriver))
						$result=true;
				}
				
				return ($result);
			}
		}
		
		/**
     * PDO Bind Param with :namespace
     * 
     * @param array $array
    **/
    private function _bindPdoNameSpace($array=array()) {
      if (strstr(key($array),' ')) {
        /* bind array data in PDO */
        foreach ($array as $f => $v) {
          /* get table column from array key */
          $field=$this->getFieldFromArrayKey($f);
          /* check pass data type for appropriate field */
          switch (gettype($array[$f])):
            /* is string found then PDO param as string */
            case 'string':
              $this->_oSTH->bindParam(":s"."_"."$field",$array[$f],PDO::PARAM_STR);
              break;
            /* if int found then PDO param set as int */
            case 'integer':
              $this->_oSTH->bindParam(":s"."_"."$field",$array[$f],PDO::PARAM_INT);
              break;
            /* if boolean found then set PDO param as boolean */
            case 'boolean':
              $this->_oSTH->bindParam(":s"."_"."$field",$array[$f],PDO::PARAM_BOOL);
              break;
          endswitch;
        }
      }
      else {
        /* bind array data in PDO */
        foreach ($array as $f => $v) {
          /* check pass data type for appropriate field */
          switch (gettype($array[$f])):
            /* is string found then PDO param as string */
            case 'string':
              $this->_oSTH->bindParam(":s"."_"."$f",$array[$f],PDO::PARAM_STR);
              break;
            /* if int found then PDO param set as int */
            case 'integer':
              $this->_oSTH->bindParam(":s"."_"."$f",$array[$f],PDO::PARAM_INT);
              break;
            /* if boolean found then set PDO param as boolean */
            case 'boolean':
              $this->_oSTH->bindParam(":s"."_"."$f",$array[$f],PDO::PARAM_BOOL);
              break;
          endswitch;
        }
      }
    }
		
		/**
     * Bind PDO Param without :namespace
     * 
     * @param array $array
    **/
    private function _bindPdoParam($array=array()) {
      /* bind array data in PDO */
      foreach ($array as $f => $v) {
        /* check pass data type for appropriate field */
        switch (gettype($array[$f])):
          /* is string type found then PDO param as string */
          case 'string':
            $this->_oSTH->bindParam($f+1,$array[$f],PDO::PARAM_STR);
            break;
          /* if int type found then PDO param set as int */
          case 'integer':
            $this->_oSTH->bindParam($f+1,$array[$f],PDO::PARAM_INT);
            break;
          /* if boolean type found then set PDO param as boolean */
          case 'boolean':
            $this->_oSTH->bindParam($f+1,$array[$f],PDO::PARAM_BOOL);
            break;
        endswitch;
      }
    }
		
		/**
     * Catch Error in txt file
     *
     * @param mixed $msg
    **/
    private function Error($msg,$isManifest=false) {
      /* log set as true */
      if ($this->log) {
        if ($isManifest) {
          /* show executed query with error */
          $this->showQuery();
          /* die code */
          $this->manifest()->errorBox($msg);
        }
      }
      else {
        /* show error message in log file */
        file_put_contents(self::ERROR_LOG_FILE,date( 'Y-m-d h:m:s' ).' :: '.$msg."\n",FILE_APPEND);
        /* die with user message */
        if ($isManifest)
          $this->manifest()->Error();
      }
    }
		
		/**
     * Replaces any parameter placeholders in a query with the value of that
     * parameter. Useful for debugging. Assumes anonymous parameters from
     *
     * @return mixed
    **/
    private function interpolateQuery() {
      $sql=$this->_oSTH->queryString;
      
      /* handle insert batch data */
      if (!$this->batch) {
        $params=((is_array($this->aData)) && (count($this->aData)>0)) ? $this->aData : $this->sSql;
        if (is_array($params)) {
          /* build a regular expression for each parameter */
          foreach ($params as $key => $value) {
            if (strstr($key,' ')) {
              $real_key=$this->getFieldFromArrayKey($key);
              /* update param value with quotes, if string value */
              $params[$key]=is_string($value) ? '"'.$value.'"' : $value;
              /* make replace array */
              $keys[]=is_string($real_key) ? '/:s_'.$real_key.'/' : '/[?]/';
            }
            else {
              /* update param value with quotes, if string value */
              $params[$key]=is_string($value) ? '"'.$value.'"' : $value;
              /* make replace array */
              $keys[]=is_string( $key ) ? '/:s_'.$key.'/' : '/[?]/';
            }
          }
          
          $sql=preg_replace($keys,$params,$sql,1,$count);
          if (strstr($sql,':s_')) {
            foreach ($this->aWhere as $key => $value) {
              if (strstr($key,' ' )) {
                $real_key=$this->getFieldFromArrayKey($key);
                /* update param value with quotes, if string value */
                $params[$key]=is_string($value) ? '"'.$value.'"' : $value;
                /* make replace array */
                $keys[]=is_string($real_key) ? '/:s_'.$real_key.'/' : '/[?]/';
              }
              else {
                /* update param value with quotes, if string value */
                $params[$key]=is_string($value) ? '"'.$value.'"' : $value;
                /* make replace array */
                $keys[]=is_string($key) ? '/:s_'.$key.'/' : '/[?]/';
              }
            }
            
            $sql=preg_replace($keys,$params,$sql,1,$count);
          }
          
          return ($sql);
        }
        else {
          return ($params);
        }
      }
      else {
        $params_batch=((is_array($this->aData)) && (count($this->aData)>0)) ? $this->aData : $this->sSql;
        $batch_query='';
        
        if (is_array($params_batch)) {
          /* build a regular expression for each parameter */
          foreach ($params_batch as $keys => $params) {
						foreach ($params as $key => $value) {
              if (strstr($key,' ')) {
                $real_key=$this->getFieldFromArrayKey($key);
                /* update param value with quotes, if string value */
                $params[$key]=is_string($value) ? '"'.$value.'"' : $value;
                /* make replace array */
                $array_keys[]=is_string($real_key) ? '/:s_'.$real_key.'/' : '/[?]/';
              }
              else {
                /* update param value with quotes, if string value */
                $params[$key]=is_string($value) ? '"'.$value.'"' : $value;
                /* make replace array */
                $array_keys[]=is_string($key) ? '/:s_'.$key.'/' : '/[?]/';
              }
            }
            
            $batch_query.='<br />'.preg_replace($array_keys,$params,$sql,1,$count);
          }
          
          return ($batch_query);
        }
        else {
          return ($params_batch);
        }
      }
    }
		
		/**
		 * Custom WHERE condition
     * @param array $array_data
     * 
     * @return array
    **/
    private function customWhere($array_data=array()) {
      $syntax='';
			
      foreach ($array_data as $key => $value) {
        $key=trim($key);
        
        /* get the first space */
        if (strstr($key,' ')) {
          /* split the current string by space */
          $array=explode(' ',$key);
          
          if (count($array)=='2') {
            $random='';
            $field=$array[0];
            $operator=$array[1];
            $tmp[]="$field $operator :s_$field"."$random";
            $syntax.=" $field $operator :s_$field"."$random ";
          }
          else if (count($array)=='3') {
            $random='';
            $condition=$array[0];
            $field=$array[1];
            $operator=$array[2];
            $tmp[]="$condition $field $operator :s_$field"."$random";
            $syntax.=" $condition $field $operator :s_$field"."$random ";
          }
        }
      }
      
      return (array('where' => $syntax,'bind' => implode(' ',$tmp)));
    }
		
		/**
		 *
		**/
		private function Debug() {
			if (!empty($this->errorCallbackFunction)) {
				$error=array("Error" => $this->strError);
				
				if (!empty($this->sSql))
					$error["SQL Statement"]=$this->sSql;
				
				if (!empty($this->bind))
					$error["Bind Parameters"]=trim(print_r($this->bind,true));
				
				$backtrace=debug_backtrace();
				if (!empty($backtrace)) {
					foreach ($backtrace as $info) {
						if ($info["file"]!=__FILE__)
							$error["Backtrace"]=$info["file"]." at line ".$info["line"];
					}
				}
				
				$msg='';
				if ($this->errorMsgFormat=='html') {
					if (!empty($error["Bind Parameters"]))
						$error["Bind Parameters"]="<pre>".$error["Bind Parameters"]."</pre>";
					
					$css=trim(file_get_contents(dirname(__FILE__)."/_css/error.css"));
					$msg.='<style type="text/css">'."<br />".$css . "<br /></style>";
					$msg.="<br />".'<div class="db-error">'."<br />\t<h3>SQL Error</h3>";
					
					foreach($error as $key => $val)
						$msg.= "<br />\t<label>".$key.":</label>".$val;
					
					$msg.="<br />\t</div><br /></div>";
				}
				else if ($this->errorMsgFormat=="text") {
					$msg.="SQL Error\n".str_repeat("-", 50);
					
					foreach($error as $key => $val)
						$msg.="\n\n$key:\n$val";
				}
				
				$func=$this->errorCallbackFunction;
				$func($msg);
			}
		}
    
    /**
     * Create an array
     *
     * @return array
    **/
    private function Cleanup($bind) {
      if (!is_array($bind)) {
				if (!empty($bind))
					$bind=array($bind);
				else
					$bind=array();
			}
			
			return ($bind);
		}
    
    /**
     * Get all information of the current RDBMS
     *
     * @return string
    **/
    private function RDBMSInfo() {
      $attributes=array("AUTOCOMMIT","ERRMODE","CASE","CLIENT_VERSION","CONNECTION_STATUS",
                        "ORACLE_NULLS","PERSISTENT","PREFETCH","SERVER_INFO","SERVER_VERSION","TIMEOUT");
      
      foreach ($attributes as $val) {
        echo "PDO::ATTR_$val: ";
        $res=$this->getAttribute(constant("PDO::ATTR_$val"));
        $pos=strpos($res,null);
        if ($pos===false) {
          echo  $res."<br />";
        }
      }
      die();
    }
    
    /**
     * Get the name of fields of a table
     *
     * @return array
    **/
    private function Filter($table,$info) {
			$driver=$this->getAttribute(PDO::ATTR_DRIVER_NAME);
			
      if ($driver=='sqlite') {
				$sql="PRAGMA table_info('".$table."');";
				$key="name";
			}
			/*else if ($driver=='mysql') {
				$sql="DESCRIBE ".$table.";";
				$key="Field";
			}*/
			else {	
        $sql="SELECT column_name FROM information_schema.columns WHERE table_name = '".$table."';";
				$key="column_name";
			}
      
			if (false!==($list=$this->ExecuteCommand($sql))) {
        $fields=array();
				
				foreach ($list as $record)
					$fields[]=$record[$key];
        
				return (array_values(array_intersect($fields, array_keys($info))));
			}
			
			return (array());
		}
		
    /************************************/
    /******* END: PRIVATE METHODS *******/
    /************************************/
    
    /******************************/
    /******* PUBLIC METHODS *******/
    /******************************/
    
		/**
     * get a new instance of the PDOToolManifest class
     *
     * @return PDOToolManifest
    **/
    public function Manifest() {
      return (new PDOToolManifest());
    }
    
    /**
     * get instance of the PDO class as singleton pattern
     *
     * @param array $dsn
     *
     * @return object $oPDO
    **/
    public static function getPDO($dsn=array()) {
      /* if not set self PDO object property or PDO set as null */
      if (!isset(self::$oPDO) || (self::$oPDO!==null)) {
        /* set class PDO property with new connection */
        self::$oPDO = new self($dsn);
      }
      
      /* return class property object */
      return (self::$oPDO);
    }
    
    /**
     * set the error message command and output format
     *
     * @param command $errorCallbackFunction output_format $errorMsgFormat
    **/
    public function setErrorCallbackFunction($errorCallbackFunction,$errorMsgFormat='html') {
			/**
			 * Variable functions for won't work with language constructs
			 * such as echo and print, so these are replaced with print_r
			**/
			if (in_array(strtolower($errorCallbackFunction),array("echo","print")))
				$errorCallbackFunction='print_r';
			
			if (function_exists($errorCallbackFunction)) {
				$this->errorCallbackFunction = $errorCallbackFunction;	
				if (!in_array(strtolower($errorMsgFormat),array("html","text")))
					$errorMsgFormat="html";
				$this->errorMsgFormat=$errorMsgFormat;	
			}	
		}
		
		/**
     * Set PDO Error Mode to get an error log file or true to show error on screen
     *
     * @param bool $mode
    **/
    public function setErrorLog($mode=false) {
      $this->log=$mode;
    }
		
		/**
     * Show executed query on call
     * 
     * @param boolean $logfile set true if wanna log all query in file
     * 
     * @return PdoWrapper
    **/
    public function showQuery($logfile=false) {
      if (!$logfile) {
        echo "<div style='color:#990099; border:1px solid #777; padding:2px; background-color: #E5E5E5;'>";
        echo " Executed Query -> <span style='color:#008000;'> ";
        echo $this->manifest()->formatSQL($this->interpolateQuery());
        echo "</span></div>";
        
        return ($this);
      }
      else {
        /* show error message in log file */
        file_put_contents(self::SQL_LOG_FILE,date('Y-m-d h:m:s').' :: '.$this->interpolateQuery()."\n", FILE_APPEND );
        return ($this);
      }
    }
		
		/**
     * Return PDO Query results array/json/xml type
     *
     * @param string $type
     *
     * @return mixed
    **/
    public function Results($type='array') {
      switch ($type) {
        case 'array':
          /* return array data */
          return $this->aResults;
          break;
        case 'xml':
          /* send the xml header to the browser */
          header( "Content-Type:text/xml" );
          /* return xml content */
          return $this->manifest()->arrayToXml($this->aResults);
          break;
        case 'json':
          /* set header as json */
          header( 'Content-type: application/json; charset="utf-8"' );
          /* return json encoded data */
          return json_encode($this->aResults);
          break;
      }
    }
		
		/**
     * Get Total Number Of Records in Requested Table
     *
     * @param string $sTable
     * @param string $where
     * @return number
    **/
    public function Count($sTable='',$sWhere='') {
      /* check if table name is not empty */
      if (!empty($sTable)) {
        /* if there is no WHERE condition */
				if (empty($sWhere)) {
          /* create count query */
          $this->sSql="SELECT COUNT(*) AS NUMROWS FROM `$sTable`;";
        }
        /* if there is a WHERE condition */
				else {
          /* create count query */
          $this->sSql="SELECT COUNT(*) AS NUMROWS FROM `$sTable` WHERE $sWhere;";
        }
        /* PDO prepare statement */
        $this->_oSTH=$this->prepare($this->sSql);
        
        try {
          if ($this->_oSTH->execute()) {
            /* fetch array result */
            $this->aResults=$this->_oSTH->fetch();
            /* close PDO */
            $this->_oSTH->closeCursor();
            
            /* return number of count */
            return ($this->aResults['NUMROWS']);
          }
          else {
            self::Error($this->_oSTH->errorInfo());
          }
        }
        catch (PDOException $e) {
          self::Error($e->getMessage().': '. __LINE__ );
        }
      }
      else {
        self::Error(self::$error_messages[5]);
      }
    }
		
		/**
     * Get the Last Insert id by Insert query
     *
     * @return number
    **/
    public function getLastInsertId() {
      return ($this->iLastId);
    }
		
		/**
     * Get the Affected rows by PDO Statement
     *
     * @return number:boolean
    **/
    public function affectedRows() {
      return (is_numeric($this->iAffectedRows) ? $this->iAffectedRows : false);
    }
    
    /**
		 * ExecuteCommand - run the current SQL command
		 *
		 * @return with an array or a row count or boolean value
		*/
    public function ExecuteCommand($sql,$bind='') {
      $this->sSql=trim($sql);
			$this->bind=$this->Cleanup($bind);
			$this->strError='';
			
			try {
				$pdostmt=$this->prepare($this->sSql);
				if ($pdostmt->execute($this->bind)!==false) {
					if (preg_match("/^(".implode("|",array("SELECT","DESCRIBE","PRAGMA")).")/i",$this->sSql)) {
            $this->iAffectedRows=$pdostmt->rowCount();
            return ($pdostmt->fetchAll(PDO::FETCH_ASSOC));
					}
					else if (preg_match("/^(".implode("|", array("DELETE","INSERT","UPDATE","REPLACE")).")/i",$this->sSql))
						return ($pdostmt->rowCount());
				}
			}
			catch (PDOException $e) {
				$this->strError=$e->getMessage();	
				$this->debug();
				
				return (false);
			}
		}
    
    /**
		 * NumRows - get the number of rows in a result: PDO Tool Class (PTC)
		*/
		public function NumRowsPTC() {
      return ($this->iAffectedRows);
		}
    
    /**
     * Execute SELECT command: PDO Tool Class (PTC)
    */
    public function SelectPTC($table,$filter='',$bind='',$fields='*') {
			$sql="SELECT ".$fields." FROM ".$table;
			
			//die($sql."; ".$table."; ".$filter."; ".$bind."; ".$fields);
			
      if (!empty($filter))
				$sql.=" ".$filter;
			$sql.=';';
      
      return ($this->ExecuteCommand($sql,$bind));
		}
    
    /**
     * Execute INSERT command: PDO Tool Class (PTC)
    */
    public function InsertPTC($table,$info,$replace=false) {
			$fields=$this->Filter($table,$info);
			if ($replace)
        $sql="REPLACE INTO ".$table." (".implode($fields,", ").") VALUES (:".implode($fields,", :").");";
      else
        $sql="INSERT INTO ".$table." (".implode($fields,", ").") VALUES (:".implode($fields,", :").");";
			$bind=array();
			
      foreach($fields as $field)
				$bind[":$field"]=$info[$field];
			
			return ($this->ExecuteCommand($sql,$bind));
		}
    
    /**
     * Execute UPDATE command: PDO Tool Class (PTC)
    */
    public function UpdatePTC($table,$info,$where,$bind='') {
      $fields=$this->Filter($table,$info);
			$fieldSize=sizeof($fields);
      
      $sql="UPDATE ".$table." SET ";
			for ($f=0;$f<$fieldSize;++$f) {
				if ($f>0)
					$sql.=', ';
        
        $sql.=$fields[$f]." = :update_".$fields[$f];
        
			}
			$sql.=" WHERE ".$where.";";
			
      $bind=$this->Cleanup($bind);
			foreach($fields as $field) {
				$bind[":update_$field"]=$info[$field];
      }
      
			return ($this->ExecuteCommand($sql,$bind));
		}
    
    /**
     * Execute DELETE command: PDO Tool Class (PTC)
    **/
    public function DeletePTC($table,$where,$bind='') {
			$sql="DELETE FROM ".$table." WHERE ".$where.";";
			
      return ($this->ExecuteCommand($sql,$bind));
		}
		
    /**
		 * FetchAssoc - fetch a result row as an associative array
		*/
		public function FetchAssocPTC($result) {
      if ($result==='')
				die('The set of result cannot be empty!');
      
      return ($result->fetch(PDO::FETCH_ASSOC));
    }
    
    /**
		 * FetchAssoc - fetch a result row as an associative array
		*/
		public function FetchArrayPTC($result,$fetch=PDO::FETCH_BOTH) {
      if ($result==='')
				die('The set of result cannot be empty!');
      
      return ($result->fetchAll($fetch));
    }
    
		/**
     * Execute PDO SELECT Query/Statement command
     *
     * @param string $sTable
     * @param array $aColumn
     * @param array $aWhere
     * @param string $sOther
     *
     * @return multi type: array/error
    **/
    public function Select($sTable='',$aColumn=array(),$aWhere=array(),$sOther='') {
      /* handle column array data */
      if (!is_array($aColumn))
        $aColumn=array();
      
      /* get field if pass otherwise use * */
      $sField=count($aColumn)>0 ? implode(', ',$aColumn) : '*';
      /* check if table name is not empty */
      if (!empty($sTable)) {
        /* if more then 0 array found in where array */
        if (count($aWhere)>0 && is_array($aWhere)) {
          /* set class where array */
          $this->aData=$aWhere;
          /* parse where array and get in temp var with key name and val */
          if (strstr(key($aWhere),' ')) {
            $tmp=$this->customWhere($this->aData);
            /* get where syntax with namespace */
            $sWhere=$tmp['where'];
          }
          else {
            foreach ($aWhere as $k => $v) {
              $tmp[]="$k = :s_$k";
            }
            /* join temp array with AND condition */
            $sWhere=implode(' AND ',$tmp);
          }
          
          /* unset temp var */
          unset($tmp);
          /* set class SQL property */
          $this->sSql="SELECT $sField FROM `$sTable` WHERE $sWhere $sOther;";
        }
        else {
          /* if no where condition pass by user */
          $this->sSql="SELECT $sField FROM `$sTable` $sOther;";
        }
        
        /* PDO prepare statement with SQL query */
        $this->_oSTH=$this->prepare( $this->sSql );
        
        /* if where condition has valid array number */
        if (count($aWhere)>0 && is_array($aWhere)) {
          /* bind PDO param */
          $this->_bindPdoNameSpace($aWhere);
        }
        
        try {
          /* check if PDO execute */
          if ($this->_oSTH->execute()) {
            /* set class property with affected rows */
            $this->iAffectedRows=$this->_oSTH->rowCount();
            /* set class property with SQL result */
            $this->aResults=$this->_oSTH->fetchAll();
            /* close PDO */
            $this->_oSTH->closeCursor();
            
            /* return self object*/
            return ($this);
          }
          else {
            /* catch PDO error */
            self::Error($this->_oSTH->errorInfo());
          }
        }
        catch (PDOException $e) {
          /* get PDO error and pass on error method */
          self::Error($e->getMessage().': '. __LINE__ );
        }
      }
      else {
        self::Error(self::$error_messages[5]);
      }
    }
		
		/**
     * Execute PDO INSERT command
     *
     * @param string $sTable
     * @param array $aData
     *
     * @return number last insert ID
    **/
    public function Insert($sTable,$aData=array()) {
      /* check if table name is not empty */
      if (!empty($sTable)) {
        /* and array data not empty */
        if (count($aData)>0 && is_array($aData)) {
          /* get array insert data in temp array */
          foreach ($aData as $f => $v) {
            $tmp[]=":s_$f";
          }
          
          /* make name space param for PDO insert statement */
          $sNameSpaceParam=implode(',',$tmp);
          /* unset temp var */
          unset($tmp);
          /* get insert fields name */
          $sFields=implode(',',array_keys($aData));
          /* set PDO insert statement in class property */
          $this->sSql="INSERT INTO `$sTable` ($sFields) VALUES ($sNameSpaceParam);";
          /* the PDO prepare statement */
          $this->_oSTH=$this->prepare($this->sSql);
          /* set class where property with array data */
          $this->aData=$aData;
          /* bind PDO param */
          $this->_bindPdoNameSpace($aData);
          
          try {
            /* execute PDO statement */
            if ($this->_oSTH->execute()) {
              /* set class property with last insert id */
              $this->iLastId=$this->lastInsertId();
              /* close PDO */
              $this->_oSTH->closeCursor();
              
              /* return this object */
              return ($this);
            }
            else {
              self::Error($this->_oSTH->errorInfo());
            }
          }
          catch (PDOException $e) {
            /* get PDO error and pass on error method */
            self::Error($e->getMessage().': '. __LINE__ );
          }
        }
        else {
          self::Error(self::$error_messages[6]);
        }
      }
      else {
        self::Error(self::$error_messages[5]);
      }
    }
		
		/**
     * Execute PDO UPDATE Statement command
     * Get No OF Affected Rows updated
     *
     * @param string $sTable
     * @param array $aData
     * @param array $aWhere
     * @param string $sOther
     *
     * @return number
    **/
    public function Update($sTable='',$aData=array(),$aWhere=array(),$sOther='') {
      /* check if table name is not empty */
      if (!empty($sTable)) {
        /* check if array data and where array is more then 0 */
        if (count($aData)>0 && count($aWhere)>0) {
          /* parse array data and make a temp array */
          foreach ($aData as $k => $v) {
            $tmp[]="$k = :s_$k";
          }
          
          /* join temp array value with , */
          $sFields=implode(', ',$tmp);
          /* delete temp array from memory */
          unset($tmp);
          
          /* parse where array and store in temp array */
          foreach ($aWhere as $k => $v) {
            $tmp[]="$k = :s_$k";
          }
          
					$this->aData=$aData;
          $this->aWhere=$aWhere;
          /* join where array value with AND operator and create where condition */
          $sWhere=implode(' AND ',$tmp);
          /* unset temp array */
          unset( $tmp );
          
          /* make SQL query to update  */
          $this->sSql="UPDATE `$sTable` SET $sFields WHERE $sWhere $sOther;";
          /* on PDO prepare statement */
          $this->_oSTH=$this->prepare($this->sSql);
          /* bind PDO param for update statement */
          $this->_bindPdoNameSpace($aData);
          /* bind PDO param for where clause */
          $this->_bindPdoNameSpace($aWhere);
          
          try {
            if ($this->_oSTH->execute()) {
              /* get affected rows */
              $this->iAffectedRows = $this->_oSTH->rowCount();
              /* close PDO */
              $this->_oSTH->closeCursor();
              
              /* return self object */
              return ($this);
            }
            else {
              self::Error($this->_oSTH->errorInfo());
            }
          }
          catch (PDOException $e) {
            self::Error($e->getMessage().': '. __LINE__ );
          }
        }
        else {
          self::Error(self::$error_messages[7]);
        }
      }
      else {
        self::Error(self::$error_messages[5]);
      }
    }
    
    /**
     * Execute PDO DELETE Query
     *
     * @param string $sTable
     * @param array $aWhere
     * @param string $sOther
     *
     * @return object PDO object
    **/
    public function Delete($sTable,$aWhere=array(),$sOther='') {
      /* check if table name is not empty */
      if (!empty($sTable)) {
        /* check where condition array length */
        if (count($aWhere)>0 && is_array($aWhere)) {
          /* make a temp array from where array data */
          foreach ($aWhere as $k => $v) {
            $tmp[]="$k = :s_$k";
          }
          
          /* join array values with AND Operator */
          $sWhere=implode(' AND ',$tmp);
          /* delete temp array */
          unset($tmp);
          
          /* set DELETE PDO Statement */
          $this->sSql="DELETE FROM `$sTable` WHERE $sWhere $sOther;";
          /* Call PDO Prepare Statement */
          $this->_oSTH=$this->prepare($this->sSql);
          /* bind delete where param */
          $this->_bindPdoNameSpace($aWhere);
          /* set array data */
          $this->aData=$aWhere;
          
          try {
            if ($this->_oSTH->execute()) {
              /* get affected rows */
              $this->iAffectedRows=$this->_oSTH->rowCount();
              /* close PDO */
              $this->_oSTH->closeCursor();
              
							/* return this object */
              return ($this);
            }
            else {
              self::Error($this->_oSTH->errorInfo());
            }
          }
          catch (PDOException $e) {
            self::Error($e->getMessage().': '. __LINE__);
					}
        }
        else {
          self::Error(self::$error_messages[8]);
        }
      }
      else {
        self::Error(self::$error_messages[5]);
      }
    }
		
		/**
     * Execute a new PDO Query command
     *
     * @param string $sSql
     * @param array Bind Param Value
     *
     * @return PdoWrapper|multi type:|number
    **/
    public function pdoQuery($sSql='',$aBindWhereParam=array()) {
      /* clean query from white space */
      $sSql=trim($sSql);
      /* get operation type */
      $operation=explode(' ',$sSql);
      /* make first word in uppercase */
      $operation[0]=strtoupper($operation[0]);
      
      /* check valid SQL operation statement */
      if (!in_array($operation[0],$this->aValidOperation)) {
        self::error(self::$error_messages[3].implode(', ',$this->aValidOperation));
      }
      
      /* SQL query pass with no bind param */
      if (!empty($sSql) && count($aBindWhereParam)<=0) {
        /* set class property with pass value */
        $this->sSql=$sSql;
        /* set class statement handler */
        $this->_oSTH=$this->prepare($this->sSql);
        
        try {
          /* execute PDO statement */
          if ($this->_oSTH->execute()) {
            /* get affected rows and set it to class property */
            $this->iAffectedRows=$this->_oSTH->rowCount();
            /* set PDO result array with class property */
            $this->aResults=$this->_oSTH->fetchAll();
            /* close PDO cursor */
            $this->_oSTH->closeCursor();
            
            /* return PDO result */
            return ($this);
          }
          else {
            /* if does not run PDO statement then send an error */
            self::error($this->_oSTH->errorInfo());
          }
        }
        catch (PDOException $e) {
          self::error($e->getMessage().': '. __LINE__ );
        }
      }
      /* if SQL query pass with bind param */
      else if (!empty($sSql) && count($aBindWhereParam)>0) {
        /* set class property with pass query */
        $this->sSql=$sSql;
        /* set class where array */
        $this->aData=$aBindWhereParam;
        /* set class PDO statement handler */
        $this->_oSTH=$this->prepare($this->sSql);
        
        /* start binding fields */
        /* bind PDO param */
        $this->_bindPdoParam($aBindWhereParam);
        try {
          /* run PDO statement with bind param */
          if ($this->_oSTH->execute()) {
            /* check operation type */
            switch ($operation[0]):
              case 'SELECT':
                /* get affected rows by select statement */
                $this->iAffectedRows=$this->_oSTH->rowCount();
                /* get PDO result array */
                $this->aResults=$this->_oSTH->fetchAll();
                
                return ($this);
                break;
              case 'INSERT':
                /* return last insert id */
                $this->iLastId=$this->lastInsertId();
                
								return ($this);
                break;
              case 'UPDATE':
                /* get affected rows */
                $this->iAffectedRows=$this->_oSTH->rowCount();
                
                return ($this);
                break;
              case 'DELETE':
                /* get affected rows */
                $this->iAffectedRows = $this->_oSTH->rowCount();
                
                return ($this);
                break;
            endswitch;
            
            /* close PDO cursor */
            $this->_oSTH->closeCursor();
          }
          else {
            self::error($this->_oSTH->errorInfo());
          }
        }
        catch (PDOException $e) {
          self::error($e->getMessage().': '. __LINE__ );
        }
      }
      else {
        self::error(self::$error_messages[4]);
      }
    }
    
    /***********************************/
    /******* END: PUBLIC METHODS *******/
    /***********************************/
  }
?>
