<?php
/*
The Logger class is used to maintain a persistant log file as
efficiently as possible.  Using Logger, your programs can append log
entries to as many different files as you need, using only 1 fopen()
call and 1 fclose() call.
It's primary use is for debugging personal programs when you can't or
don't want to log to "error_log".  (That, and this class is a bit more
efficient than error_log, IMHO.
The log file you specify gets opened when you first call the log()
method with the specified log handle. The log file remains open until
your process calls the close_logs() method, which closes all currently
open log files.
*/
class LoggerTool {
	var $FILELIST=array();
	var $ROOT;
	var $MYSYSTEM;
	var $PID;
	var $FORMAT;
	var $DEBUG;
	var $ERROR;
	var $HANDLE;
	
	// constructor
	// $LogRoot => current directory path
	// $FileArray => a new array with file names(array('FileName1,FileName2,...'))
	public function __construct($LogRoot,$FileArray='') {
		$this->ROOT='';
		// current operating system
		$this->MYSYSTEM=php_uname('s');
		$this->PID=getmypid();
		$this->FORMAT="%b %d %H:%M:%S";
		$this->DEBUG=true;
		$this->ERROR='';
		$this->HANDLE=null;
		
		if (empty($LogRoot)) {
			if ($DEBUG) {
				error_log('[ErrorLog]: need LogRoot to initialize.',0);
			}
			return;
		}
		$this->SetRoot($LogRoot);
		if (!empty($FileArray)) {
			$this->initialize($FileArray);
		}
		return;
	}
	
	// init function using current array with file name(s)
	private function Initialize($FileArray) {
		// get $FileArray type - if it is not an array then send an error message
		if ((gettype($FileArray))!='array') {
			if ($DEBUG) {
				$message='[ErrorLog]: invalid data type.';
				error_log($message,0);
				$this->ERROR=$message;
			}
			return;
		}
		while (list($key,$val)=each($FileArray)) {
			if ((isset($key)) && (isset($val))) {
				$val=$this->ROOT.$val;
				// $this->FILELIST[$key] contains the file list
				$this->FILELIST[$key]=$val;
			}
		}
		return;
	}
	
	// check parameters
  // 0 - false; 1 - true
	private function Is_Sane($fileName='') {
    $exists=false;
    
		if (empty($fileName)) {
			return false;
		}
		//if ($must_exist!=0) {
			if (!file_exists($fileName)) {
				$this->ERROR='is_sane: ['.$fileName.'] does not exist.';
				return false;
			}
			$exists=true;
		//}
		if ($exists) {
			if (!is_readable($fileName)) {
				$this->ERROR='is_sane: ['.$fileName.'] is not readable.';
				return false;
			}
      if (!is_writeable($fileName)) {
        $this->ERROR='is_sane: ['.$fileName.'] is not writable.';
        return false;
      }
      //if ($noDirs!=0) {
				if (is_dir($fileName)) {
					$this->ERROR='is_sane: ['.$fileName.'] is a directory.';
					return false;
				}
			//}
			//if ($noSymLinks!=0) {
				if (is_link($fileName)) {
					$this->ERROR='is_sane: ['.$fileName.'] is a symlink.';
					return false;
				}
			//}
		}
		
		return (true);
	}
	
	// read ERROR variable and return with it
	public function Get_Message() {
		return ($this->ERROR);
	}
	
	// start log function
	// $Handle => which file we want to use from the $FileArray array
	// $LogEntry => a message we want to write into the current log file
	public function Log($Handle,$LogEntry) {
		global $php_errormsg;
		ini_set('track_errors',1);
		
		// get current file name from the current file list
		// we need the current number of the log file we want to use
		// the current means $Handle
		$filename=$this->FILELIST[$Handle];
		// first we want to create the current date and time as a timestamp
		$TimeStamp=strftime($this->FORMAT,time());
		// second we want to create the log message
		// our log_message=timestamp + current PID + our log entry
		$LogEntry=$TimeStamp.'['.$this->PID.']'.$LogEntry;
		// if the cuurent log file does not exist...
		//if (!$this->$Handle) {
		if (!$Handle) {
			// ... we need to create it
			$fd=@fopen($filename,"a");
			//echo $filename;
			// if we cannot create it we send an error message
			if ((!$this->is_sane($filename)) || (!$fd) || (empty($fd))) {
				if ($this->DEBUG) {
					$message='[ErrorLog]: Fatal file error ['.$php_errormsg.']';
					error_log($message,0);
					$this->ERROR=$message;
					return false;
				}
				return false;
			}
			// we can save our complete log message into this file
			fwrite($fd,"$LogEntry\n");
			$this->HANDLE=$fd;
		}
		// else if this file is exists
		// then we can write our log message into this file
		else {
			fwrite($this->HANDLE,"$LogEntry\n");
		}
		
		return (true);
	}
	
	// stop log function - close log file(s)
	public function CloseLogs() {
		// while it can be found log file(s) it must be closed
		while (list($Handle,$Val)=each($this->FILELIST)) {
			if ($Handle) {
				// create the last log message and write it into the log file(s)
				$TimeStamp=strftime($this->FORMAT,time());
				$LogEntry=$TimeStamp.' ['.$this->PID.'] [ErrorLog]: Closing Log File';
				// write last log message into the log file(s)
				fwrite($this->HANDLE,"$LogEntry\n");
				//fwrite($Handle,"$LogEntry\n");
				// close log file(s)
				fclose($this->HANDLE);
				unset($this->HANDLE);
			}
		}
	}
	
	// get the current log directory
	public function SetRoot($root) {
		if ($this->MYSYSTEM=='Windows NT') {
			// WIN32 box - no test
			if (!preg_match("/\\$/",$root)) {
				$root=$root.'\\';
			}
			$this->ROOT=$root;
		}
		else {
			if (!preg_match("/\/$/",$root)) {
				$root.='/';
			}
			if (is_dir($root)) {
				$this->ROOT=$root;
			}
			else {
				$this->ROOT='';
				if ($DEBUG) {
					error_log('Specified ROOT dir ['.$root.'] is not a directory.',0);
					error_log($message,0);
					$this->ERROR=$message;
					return false;
				}
			}
		}
	}
}
?>
