<?php
  /* A light and permissions-checking logging PHP class
	 *
	 * Author	: Thomas Garai
	 * Date	: November 21, 2013
	 * Comments	: Originally was written by Kenneth Katzgrau <katzgrau@gmail.com> for use with wpSearch
	 * Website	: http://www.aniko.co.hu
	 *
	 * Usage:
	 *		$log=new AniSysLog("your_filename.ext",AniSysLog::ANI_YOURLOGGINGMETHOD);
	 *		$log->LogInfo("The search method has been successfully run.");  //Prints to the log file
	 *		$log->LogFatal("Serious problem has been occured.");            //Prints to the log file
	 *		$log->LogDebug("The result is: x = 3.1415");                    //Prints nothing due to priority setting
	*/
  
  class AniSysLog {
    /* CONSTANTS */
    /* Current version */
    const ANI_VERSION=0.1;
    /* Type of any logging method */
    const ANI_DEBUG=1;		// Most Verbose
		const ANI_INFO=2;			// Only generally ANI_INFOrmations
		const ANI_WARN=3;			// Only any ANI_WARNing but no more
		const ANI_ERROR=4;		// Only any ANI_ERROR but no more
		const ANI_FATAL=5;		// Least Verbose
		const ANI_OFF=6;			// Nothing at all
		const ANI_UNKNOWN=7;	// Unknown
		/* Check whether log file could be opened and store the state */
		/* statuses: open=exists_and_is_writable; failed=exists_but_is_not_writable; closed= */
    const LOG_OPEN=1;
		const OPEN_FAILED=2;
		const LOG_CLOSED=3;
		
    /* PUBLIC MEMBERS */
		/* Current status of the log file */
    public $Log_Status=AniSysLog::LOG_CLOSED;
		/* current date and time format */ 
    /* date: 4 digits YEAR:month with leading zero:day with leading zero */
    /* time: 24 hour format without leading zero:minutes with leading zero:seconds with leading zero */
    public $DateFormat="Y-m-d G:i:s";
		/* an array to save any message */
    public $MessageQueue;
		
    /* PRIVATE MEMBERS */
		/* wheter the log object has been successfully created or not */
		private $log_object_status=false;
    /* current log file, default is 'aniSysLog.log' int the current directory */
    private $log_file='';
		/* log file size, default is 1024 kbyte */
		private $log_file_size=0;
		/* default priority level from where can be write logs, default is ANI_WARN */
    private $priority=AniSysLog::ANI_WARN;
		/* file handle */
    private $file_handle;
		
    /* Constructor */
		public function __construct($filepath='aniSysLog.log',$fileLogSize=1024000,$priority=AniSysLog::ANI_WARN) {
		  if ($priority==AniSysLog::ANI_UNKNOWN )
        return;
			
			// get the current log file path
			$this->log_file=$filepath;
			/* the log file size to get the log rotate size */
			$this->log_file_size=$fileLogSize;
			// create an empty array to save temporarly the log messages
			$this->MessageQueue=array();
			// set the log priority level
			$this->priority=$priority;
			
			/* check whether file is exists */
			if (file_exists($this->log_file)) {
				/* log rotate the file when it need to */
				AniSysLog::fileLogRotate($this->log_file,$this->log_file_size);
				/* check whether file is writable */
        if (!is_writable($this->log_file)) {
					$this->Log_Status=AniSysLog::OPEN_FAILED;
					$this->MessageQueue[]="The file exists, but could not be opened for writing. Check that appropriate permissions have been set.";
					$this->log_object_status=false;
					return (false);
				}
			}
			
      /* check whether file can be opened */
			if ($this->file_handle=fopen($this->log_file,"a")) {
				$this->Log_Status=AniSysLog::LOG_OPEN;
				$this->MessageQueue[]="The log file was opened successfully.";
			}
			else {
				$this->Log_Status=AniSysLog::OPEN_FAILED;
				$this->MessageQueue[]="The file could not be opened. Check permissions.";
				$this->log_object_status=false;
				return (false);
			}
			
			$this->log_object_status=true;
			return (true);
		}
		
    /* Destructor */
    public function __destruct() {
      /* if file handle is working */
      if ($this->file_handle)
        fclose( $this->file_handle );
		}
		
		/* PRIVATE METHODS */
		/* the log method which will save the current full log content */
    private function Log($line,$priority) {
		  /* check whether the current priority is correct: 1-6 */
      //echo $this->priority.' ';
			if ($this->priority<=$priority ) {
				/* get current status */
        $status=$this->getTimeLine($priority);
				/* save current status and the log ANI_INFOrmation into the current log file */
        $this->WriteFreeFormLine($status.' '.$line.PHP_EOL);
			}
		}
		
		/* Write the current log ANI_INFO or a standard message (in case of missing permission) into the log file */
    private function WriteFreeFormLine($line) {
			/* if log file is open and can be found any log status */
      if ($this->Log_Status==AniSysLog::LOG_OPEN && $this->priority!=AniSysLog::ANI_UNKNOWN ) {
        /* the current log file cannot be written because of missing permission */
        if (fwrite($this->file_handle,$line )===false) {
          $this->MessageQueue[] = "The file could not be written to. Check that appropriate permissions have been set.";
		    }
			}
		}
		
		/* Get the current date and time and the level ANI_INFO in words */
    private function getTimeLine($level) {
		  /* get the current date and time */
      $time=date($this->DateFormat);
      
      /* get the level ANI_INFO in words */
			switch($level) {
				case AniSysLog::ANI_INFO:
					return $time.' - ANI_INFO  -->';
				case AniSysLog::ANI_WARN:
					return $time.' - ANI_WARN  -->';				
				case AniSysLog::ANI_DEBUG:
					return $time.' - ANI_DEBUG -->';				
				case AniSysLog::ANI_ERROR:
					return $time.' - ANI_ERROR -->';
				case AniSysLog::ANI_FATAL:
					return $time.' - ANI_FATAL -->';
				case AniSysLog::ANI_OFF:
					return $time.' - ANI_OFF   -->';
				default:
					return $time.' - LOG UNKNOWN -->';
			}
		}
		
		private function CheckFileSize($fileName) {
			if ($fileName==='')
				die('The checking file name you want to get the size is empty!');
			
			return (filesize($fileName));
		}
		
		private function fileLogRotate($fileName,$logSize) {
			if ($fileName==='')
				die('The checking file name you want to get the size is empty!');
			
			/* check wheter file size is more then $logSize in kbyte */
			if (AniSysLog::CheckFileSize($fileName)>$logSize) {
				/* if $fileName.old is exists then delete $fileName.old file */
				if (file_exists($fileName.'old'))
					unlink($fileName.'old');
				
				rename($fileName,$fileName.'.old');
				if ($this->file_handle=fopen($fileName,"a")) {
					$this->Log_Status=AniSysLog::LOG_OPEN;
				}
				else {
					$this->Log_Status=AniSysLog::OPEN_FAILED;
					$this->MessageQueue[]="The file could not be opened. Check permissions.";	
				}
			}
		}
		/* END: PRIVATE METHODS */
		
		/* PUBLIC METHODS */
		/* Add a new ANI_INFO into the current log file */
    public function LogInfo($line) {
		  $this->Log($line,AniSysLog::ANI_INFO);
		}
		
    /* Add a new ANI_DEBUG into the current log file */
		public function LogDebug($line) {
			$this->Log($line,AniSysLog::ANI_DEBUG);
		}
		
		/* Add a new ANI_WARN into the current log file */
    public function LogWarn($line) {
			$this->Log($line,AniSysLog::ANI_WARN);	
		}
		
    /* Add a new ANI_ERROR into the current log file */
		public function LogError($line) {
			$this->Log($line,AniSysLog::ANI_ERROR);		
		}

		/* Add a new ANI_FATAL into the current log file */
    public function LogFatal($line) {
			$this->Log($line,AniSysLog::ANI_FATAL);
		}
		
		/* the log method which will save the current full log content */
    public function LogVer() {
		  $this->Log(AniSysLog::ANI_VERSION,AniSysLog::ANI_OFF);
		}
		
		/* the current log file size */
    public function LogFileSize() {
		  return (AniSysLog::CheckFileSize($this->log_file));
		}
		
		/* check whether the new object has been created */
		public function CheckLogObjectStatus() {
			return ($this->log_object_status);
		}
		/* END: PUBLIC METHODS */
	}
?>